/*
 * Decompiled with CFR 0.152.
 */
package au.org.ala.images.tiling;

import au.org.ala.images.tiling.DefaultZoomFactorStrategy;
import au.org.ala.images.tiling.ImageTilerConfig;
import au.org.ala.images.tiling.ImageTilerResults;
import au.org.ala.images.tiling.TileFormat;
import au.org.ala.images.tiling.TilerSink;
import au.org.ala.images.tiling.ZoomFactorStrategy;
import au.org.ala.images.util.FileByteSinkFactory;
import au.org.ala.images.util.ImageReaderUtils;
import com.google.common.io.ByteSink;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.spi.IIORegistry;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ImageTiler {
    private static final Logger log = LoggerFactory.getLogger(ImageTiler.class);
    private int _tileSize = 256;
    private int _maxColsPerStrip = 6;
    private int _ioThreadCount = 2;
    private int _maxLevelThreads = 2;
    private TileFormat _tileFormat = TileFormat.JPEG;
    private Color _tileBackgroundColor = Color.gray;
    private boolean _exceptionOccurred = false;
    private ZoomFactorStrategy _zoomFactorStrategy = new DefaultZoomFactorStrategy();

    public ImageTiler(ImageTilerConfig config) {
        if (config != null) {
            this._ioThreadCount = config.getIOThreadCount();
            this._maxLevelThreads = config.getLevelThreadCount();
            this._tileSize = config.getTileSize();
            this._maxColsPerStrip = config.getMaxColumnsPerStrip();
            this._tileFormat = config.getTileFormat();
            this._tileBackgroundColor = config.getTileBackgroundColor();
            this._zoomFactorStrategy = config.getZoomFactorStrategy();
        }
    }

    public ImageTilerResults tileImage(File imageFile, File destinationDirectory) throws IOException, InterruptedException {
        return this.tileImage(FileUtils.openInputStream((File)imageFile), new TilerSink.PathBasedTilerSink(new FileByteSinkFactory(destinationDirectory)));
    }

    public ImageTilerResults tileImage(InputStream imageInputStream, TilerSink tilerSink) throws IOException, InterruptedException {
        try {
            this._exceptionOccurred = false;
            byte[] imageBytes = IOUtils.toByteArray((InputStream)imageInputStream);
            int[] pyramid = this._zoomFactorStrategy.getZoomFactors(imageBytes);
            ExecutorService levelThreadPool = Executors.newFixedThreadPool(Math.min(pyramid.length, this._maxLevelThreads));
            ExecutorService ioThreadPool = Executors.newFixedThreadPool(this._ioThreadCount);
            for (int level = pyramid.length - 1; level >= 0; --level) {
                this.submitLevelForProcessing(level, imageBytes, pyramid, tilerSink.getLevelSink(level), levelThreadPool, ioThreadPool);
            }
            levelThreadPool.shutdown();
            levelThreadPool.awaitTermination(30L, TimeUnit.MINUTES);
            ioThreadPool.shutdown();
            ioThreadPool.awaitTermination(30L, TimeUnit.MINUTES);
            if (!this._exceptionOccurred) {
                return new ImageTilerResults(true, pyramid.length);
            }
            return new ImageTilerResults(false, 0);
        }
        catch (Throwable th) {
            log.error("Exception occurred tiling image", th);
            return new ImageTilerResults(false, 0);
        }
    }

    private void submitLevelForProcessing(int level, byte[] imageBytes, int[] pyramid, TilerSink.LevelSink levelSink, ExecutorService levelThreadPool, ExecutorService ioThreadPool) {
        int subSample = pyramid[level];
        levelThreadPool.submit(new TileImageTask(imageBytes, subSample, levelSink, ioThreadPool));
    }

    private void tileImageAtSubSampleLevel(byte[] bytes, int subsample, TilerSink.LevelSink levelSink, ExecutorService ioThreadPool) throws IOException {
        ImageReader reader = ImageReaderUtils.findCompatibleImageReader(bytes);
        if (reader != null) {
            int srcHeight = reader.getHeight(0);
            int srcWidth = reader.getWidth(0);
            int height = (int)Math.ceil((double)reader.getHeight(0) / (double)subsample);
            int rows = (int)Math.ceil((double)height / (double)this._tileSize);
            int stripWidth = this._tileSize * subsample * this._maxColsPerStrip;
            int numberOfStrips = (int)Math.ceil((double)srcWidth / (double)stripWidth);
            for (int stripIndex = 0; stripIndex < numberOfStrips; ++stripIndex) {
                int srcStripOffset = stripIndex * stripWidth;
                if (srcStripOffset > srcWidth) continue;
                Rectangle stripRect = new Rectangle(srcStripOffset, 0, stripWidth, srcHeight);
                ImageReadParam params = reader.getDefaultReadParam();
                params.setSourceRegion(stripRect);
                params.setSourceSubsampling(subsample, subsample, 0, 0);
                BufferedImage strip = reader.read(0, params);
                this.splitStripIntoTiles(strip, levelSink, rows, stripIndex, ioThreadPool);
            }
        } else {
            throw new RuntimeException("No readers found suitable for file");
        }
        reader.dispose();
    }

    private void splitStripIntoTiles(BufferedImage strip, TilerSink.LevelSink levelSink, int rows, int stripIndex, ExecutorService ioThreadPool) {
        for (int col = 0; col < this._maxColsPerStrip; ++col) {
            int stripColOffset = col * this._tileSize;
            if (stripColOffset >= strip.getWidth()) continue;
            TilerSink.ColumnSink columnSink = levelSink.getColumnSink(col, stripIndex, this._maxColsPerStrip);
            int tw = this._tileSize;
            if (tw + col * this._tileSize > strip.getWidth()) {
                tw = strip.getWidth() - col * this._tileSize;
            }
            if (tw > strip.getWidth()) {
                tw = strip.getWidth();
            }
            for (int y = 0; y < rows; ++y) {
                int th = this._tileSize;
                int rowOffset = strip.getHeight() - (rows - y) * this._tileSize;
                if (rowOffset < 0) {
                    th = strip.getHeight() - (rows - 1) * this._tileSize;
                    rowOffset = 0;
                }
                BufferedImage tile = null;
                if (tw > 0 && th > 0) {
                    tile = strip.getSubimage(stripColOffset, rowOffset, tw, th);
                }
                BufferedImage destTile = this._tileFormat == TileFormat.PNG ? new BufferedImage(this._tileSize, this._tileSize, 6) : new BufferedImage(this._tileSize, this._tileSize, 5);
                Graphics g = destTile.getGraphics();
                if (this._tileFormat == TileFormat.JPEG && tile != null) {
                    g.setColor(this._tileBackgroundColor);
                    g.fillRect(0, 0, this._tileSize, this._tileSize);
                }
                if (tile != null) {
                    g.drawImage(tile, 0, this._tileSize - tile.getHeight(), null);
                }
                g.dispose();
                ByteSink tileSink = columnSink.getTileSink(rows - y - 1);
                ioThreadPool.submit(new SaveTileTask(tileSink, destTile));
            }
        }
    }

    static {
        ImageIO.scanForPlugins();
        IIORegistry.getDefaultInstance();
        ImageIO.setUseCache(false);
    }

    class TileImageTask
    implements Runnable {
        private byte[] _bytes;
        private int _subSample;
        private TilerSink.LevelSink _levelSink;
        private ExecutorService _ioThreadPool;

        public TileImageTask(byte[] bytes, int subSample, TilerSink.LevelSink levelSink, ExecutorService ioThreadPool) {
            this._bytes = bytes;
            this._subSample = subSample;
            this._levelSink = levelSink;
            this._ioThreadPool = ioThreadPool;
        }

        @Override
        public void run() {
            try {
                ImageTiler.this.tileImageAtSubSampleLevel(this._bytes, this._subSample, this._levelSink, this._ioThreadPool);
            }
            catch (Error | Exception ex) {
                ImageTiler.this._exceptionOccurred = true;
                log.error("Exception occurred during tiling image task", ex);
            }
        }
    }

    class SaveTileTask
    implements Runnable {
        protected ByteSink tileSink;
        protected BufferedImage image;

        public SaveTileTask(ByteSink tileSink, BufferedImage image) {
            this.tileSink = tileSink;
            this.image = image;
        }

        @Override
        public void run() {
            try {
                String format = ImageTiler.this._tileFormat == TileFormat.PNG ? "png" : "jpeg";
                try (OutputStream tileStream = this.tileSink.openStream();){
                    if (!ImageIO.write((RenderedImage)this.image, format, tileStream)) {
                        ImageTiler.this._exceptionOccurred = true;
                    }
                }
            }
            catch (Error | Exception ex) {
                ImageTiler.this._exceptionOccurred = true;
                log.error("Exception occurred saving file task", ex);
            }
        }
    }
}

