/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.dwc;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOCase;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.gbif.dwc.Archive;
import org.gbif.dwc.ArchiveField;
import org.gbif.dwc.ArchiveFile;
import org.gbif.dwc.DwcLayout;
import org.gbif.dwc.UnsupportedArchiveException;
import org.gbif.dwc.meta.DwcMetaFiles;
import org.gbif.dwc.terms.DcTerm;
import org.gbif.dwc.terms.DwcTerm;
import org.gbif.dwc.terms.Term;
import org.gbif.dwc.terms.TermFactory;
import org.gbif.dwcaio.shaded.com.google.common.base.Preconditions;
import org.gbif.dwcaio.shaded.com.google.common.collect.ImmutableList;
import org.gbif.utils.file.CompressionUtil;
import org.gbif.utils.file.tabular.TabularDataFileReader;
import org.gbif.utils.file.tabular.TabularFileMetadata;
import org.gbif.utils.file.tabular.TabularFileMetadataExtractor;
import org.gbif.utils.file.tabular.TabularFiles;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

class InternalDwcFileFactory {
    private static final Logger LOG = LoggerFactory.getLogger(InternalDwcFileFactory.class);
    private static final TermFactory TERM_FACTORY = TermFactory.instance();
    private static final String DEFAULT_ENDLINE_CHAR = "\n";
    private static final char DEFAULT_DELIMITER_CHAR = ',';
    private static final Map<Term, Term> TERM_TO_ROW_TYPE;
    private static final List<Term> ID_TERMS;
    private static final List<String> DATA_FILE_SUFFICES;

    private InternalDwcFileFactory() {
    }

    private static void applyIpt205Patch(Path dwcFolder) {
        Iterator iter = FileUtils.iterateFiles((File)dwcFolder.toFile(), (String[])new String[]{"xml", "txt"}, (boolean)false);
        while (iter.hasNext()) {
            File f = (File)iter.next();
            if (!f.getName().startsWith("\\")) continue;
            String orig = f.getName();
            String replacement = f.getName().replaceFirst("\\\\", "");
            LOG.info("Renaming file from {} to {}", (Object)orig, (Object)replacement);
            f.renameTo(new File(dwcFolder.toFile(), replacement));
        }
    }

    private static List<File> extractPossibleDataFile(File dwcFolder) {
        ArrayList<File> dataFiles = new ArrayList<File>();
        for (String suffix : DATA_FILE_SUFFICES) {
            IOFileFilter ff = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)suffix, (IOCase)IOCase.INSENSITIVE), HiddenFileFilter.VISIBLE});
            dataFiles.addAll(Arrays.asList(dwcFolder.listFiles((FileFilter)ff)));
        }
        return dataFiles;
    }

    private static Archive archiveFromSingleFile(Path dwcFile) throws IOException {
        Archive archive = new Archive();
        archive.setLocation(dwcFile.toFile());
        archive.setDwcLayout(DwcLayout.fromFile(dwcFile.toFile()));
        Optional<String> possibleMetadataFile = DwcMetaFiles.discoverMetadataFile(dwcFile);
        if (possibleMetadataFile.isPresent()) {
            archive.setMetadataLocation(possibleMetadataFile.get());
        } else {
            ArchiveFile coreFile = InternalDwcFileFactory.fromSingleFile(dwcFile);
            coreFile.addLocation(dwcFile.getFileName().toString());
            archive.setCore(coreFile);
        }
        return archive;
    }

    static Archive fromLocation(Path dwcLocation) throws IOException, UnsupportedArchiveException {
        if (!Files.exists(dwcLocation, new LinkOption[0])) {
            throw new FileNotFoundException("dwcLocation does not exist: " + dwcLocation.toAbsolutePath());
        }
        if (Files.isRegularFile(dwcLocation, new LinkOption[0])) {
            return InternalDwcFileFactory.archiveFromSingleFile(dwcLocation);
        }
        Archive archive = new Archive();
        InternalDwcFileFactory.applyIpt205Patch(dwcLocation);
        Path metaDescriptorFile = dwcLocation.resolve("meta.xml");
        if (Files.exists(metaDescriptorFile, new LinkOption[0])) {
            try {
                archive = DwcMetaFiles.fromMetaDescriptor(new FileInputStream(metaDescriptorFile.toFile()));
            }
            catch (IOException | SAXException e) {
                throw new UnsupportedArchiveException(e);
            }
        } else {
            List<File> dataFiles = InternalDwcFileFactory.extractPossibleDataFile(dwcLocation.toFile());
            if (dataFiles.size() == 1) {
                archive = InternalDwcFileFactory.archiveFromSingleFile(dataFiles.get(0).toPath());
            }
        }
        DwcMetaFiles.discoverMetadataFile(dwcLocation).ifPresent(archive::setMetadataLocation);
        archive.setLocation(dwcLocation.toFile());
        archive.setDwcLayout(DwcLayout.DIRECTORY_ROOT);
        return archive;
    }

    static Archive fromCompressed(Path dwcaLocation, Path destination) throws IOException, UnsupportedArchiveException {
        if (!Files.exists(dwcaLocation, new LinkOption[0])) {
            throw new FileNotFoundException("dwcaLocation does not exist: " + dwcaLocation.toAbsolutePath());
        }
        if (Files.exists(destination, new LinkOption[0])) {
            LOG.debug("Deleting existing archive folder [{}]", (Object)destination.toAbsolutePath());
            org.gbif.utils.file.FileUtils.deleteDirectoryRecursively((File)destination.toFile());
        }
        FileUtils.forceMkdir((File)destination.toFile());
        try {
            File root;
            CompressionUtil.decompressFile((File)destination.toFile(), (File)dwcaLocation.toFile(), (boolean)true);
            File[] rootFiles = destination.toFile().listFiles((FileFilter)HiddenFileFilter.VISIBLE);
            if (rootFiles.length == 1 && (root = rootFiles[0]).isDirectory()) {
                LOG.debug("Removing single root folder {} found in decompressed archive", (Object)root.getAbsoluteFile());
                for (File f : FileUtils.listFiles((File)root, (IOFileFilter)TrueFileFilter.TRUE, null)) {
                    File f2 = new File(destination.toFile(), f.getName());
                    f.renameTo(f2);
                }
            }
            return InternalDwcFileFactory.fromLocation(destination);
        }
        catch (CompressionUtil.UnsupportedCompressionType e) {
            throw new UnsupportedArchiveException((Exception)((Object)e));
        }
    }

    static ArchiveFile fromSingleFile(Path dataFile) throws UnsupportedArchiveException, IOException {
        List headers;
        Preconditions.checkArgument(Files.isRegularFile(dataFile, new LinkOption[0]), "dataFile shall be a file");
        ArchiveFile dwcFile = new ArchiveFile();
        dwcFile.setIgnoreHeaderLines(1);
        TabularFileMetadata tabularFileMetadata = TabularFileMetadataExtractor.extractTabularFileMetadata((Path)dataFile);
        dwcFile.setFieldsTerminatedBy(Optional.ofNullable(tabularFileMetadata.getDelimiter()).orElse(Character.valueOf(',')).toString());
        dwcFile.setFieldsEnclosedBy(tabularFileMetadata.getQuotedBy());
        dwcFile.setEncoding(tabularFileMetadata.getEncoding().name());
        TabularDataFileReader reader = TabularFiles.newTabularFileReader((Reader)Files.newBufferedReader(dataFile, tabularFileMetadata.getEncoding()), (char)dwcFile.getFieldsTerminatedBy().charAt(0), (String)DEFAULT_ENDLINE_CHAR, (Character)dwcFile.getFieldsEnclosedBy(), (boolean)true);
        Object object = null;
        try {
            headers = reader.getHeaderLine() == null ? Collections.emptyList() : reader.getHeaderLine();
        }
        catch (Throwable throwable) {
            object = throwable;
            throw throwable;
        }
        finally {
            if (reader != null) {
                if (object != null) {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable) {
                        ((Throwable)object).addSuppressed(throwable);
                    }
                } else {
                    reader.close();
                }
            }
        }
        int index = 0;
        for (String head : headers) {
            if (head != null && head.length() > 1) {
                try {
                    Term dt = TERM_FACTORY.findPropertyTerm(head);
                    dwcFile.addField(new ArchiveField(index, dt, null, ArchiveField.DataType.string));
                }
                catch (IllegalArgumentException e) {
                    LOG.warn("Illegal term name >>{}<< found in header, ignore column {}", (Object)head, (Object)index);
                }
            }
            ++index;
        }
        List<Term> headerAsTerm = dwcFile.getFields().keySet().stream().collect(Collectors.toList());
        InternalDwcFileFactory.determineRecordIdentifier(headerAsTerm).ifPresent(t -> dwcFile.setId(dwcFile.getField((Term)t)));
        InternalDwcFileFactory.determineRowType(headerAsTerm).ifPresent(dwcFile::setRowType);
        return dwcFile;
    }

    static Optional<Term> determineRowType(List<Term> terms) {
        return TERM_TO_ROW_TYPE.entrySet().stream().filter(ke -> terms.contains(ke.getKey())).map(Map.Entry::getValue).findFirst();
    }

    static Optional<Term> determineRecordIdentifier(List<Term> terms) {
        return ID_TERMS.stream().filter(terms::contains).findFirst();
    }

    static {
        LinkedHashMap<DwcTerm, DwcTerm> idToRowType = new LinkedHashMap<DwcTerm, DwcTerm>();
        idToRowType.put(DwcTerm.occurrenceID, DwcTerm.Occurrence);
        idToRowType.put(DwcTerm.taxonID, DwcTerm.Taxon);
        idToRowType.put(DwcTerm.eventID, DwcTerm.Event);
        TERM_TO_ROW_TYPE = Collections.unmodifiableMap(idToRowType);
        ID_TERMS = Collections.unmodifiableList(Arrays.asList(DwcTerm.occurrenceID, DwcTerm.taxonID, DwcTerm.eventID, DcTerm.identifier));
        DATA_FILE_SUFFICES = ImmutableList.of(".csv", ".txt", ".tsv", ".tab", ".text", ".data", ".dwca");
    }
}

