/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.crawler;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import java.net.URI;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.http.HttpResponse;
import org.gbif.crawler.CrawlClient;
import org.gbif.crawler.CrawlContext;
import org.gbif.crawler.CrawlListener;
import org.gbif.crawler.CrawlStrategy;
import org.gbif.crawler.RequestHandler;
import org.gbif.crawler.ResponseHandler;
import org.gbif.crawler.RetryPolicy;
import org.gbif.crawler.client.HttpCrawlClient;
import org.gbif.crawler.exception.FatalCrawlException;
import org.gbif.crawler.exception.ProtocolException;
import org.gbif.crawler.exception.TransportException;
import org.gbif.crawler.protocol.biocase.BiocaseCrawlConfiguration;
import org.gbif.crawler.protocol.biocase.BiocaseResponseHandler;
import org.gbif.crawler.protocol.biocase.BiocaseScientificNameRangeRequestHandler;
import org.gbif.crawler.protocol.digir.DigirCrawlConfiguration;
import org.gbif.crawler.protocol.digir.DigirResponseHandler;
import org.gbif.crawler.protocol.digir.DigirScientificNameRangeRequestHandler;
import org.gbif.crawler.protocol.tapir.TapirCrawlConfiguration;
import org.gbif.crawler.protocol.tapir.TapirResponseHandler;
import org.gbif.crawler.protocol.tapir.TapirScientificNameRangeRequestHandler;
import org.gbif.crawler.retry.LimitedRetryPolicy;
import org.gbif.crawler.strategy.ScientificNameRangeCrawlContext;
import org.gbif.crawler.strategy.ScientificNameRangeStrategy;
import org.gbif.wrangler.lock.Lock;
import org.gbif.wrangler.lock.NoLockFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class Crawler<CTX extends CrawlContext, REQ, RESP, RES> {
    private static final Logger LOG = LoggerFactory.getLogger(Crawler.class);
    private final CrawlStrategy<CTX> strategy;
    private final RequestHandler<CTX, REQ> requestHandler;
    private final ResponseHandler<RESP, RES> responseHandler;
    private final CrawlClient<REQ, RESP> client;
    private final RetryPolicy retryPolicy;
    private final Lock lock;
    private final Stopwatch stopwatch = Stopwatch.createUnstarted();
    private final List<CrawlListener<CTX, REQ, RES>> listeners = Lists.newArrayList();

    public Crawler(CrawlStrategy<CTX> strategy, RequestHandler<CTX, REQ> requestHandler, ResponseHandler<RESP, RES> responseHandler, CrawlClient<REQ, RESP> client, RetryPolicy retryPolicy, Lock lock) {
        this.strategy = (CrawlStrategy)Preconditions.checkNotNull(strategy, (Object)"strategy can't be null");
        this.requestHandler = (RequestHandler)Preconditions.checkNotNull(requestHandler, (Object)"requestHandler can't be null");
        this.responseHandler = (ResponseHandler)Preconditions.checkNotNull(responseHandler, (Object)"responseHandler can't be null");
        this.client = (CrawlClient)Preconditions.checkNotNull(client, (Object)"client can't be null");
        this.retryPolicy = (RetryPolicy)Preconditions.checkNotNull((Object)retryPolicy, (Object)"retryPolicy can't be null");
        this.lock = (Lock)Preconditions.checkNotNull((Object)lock, (Object)"lock can't be null");
    }

    public static Crawler<ScientificNameRangeCrawlContext, String, HttpResponse, List<Byte>> newBiocaseCrawler(UUID datasetKey, int attempt, URI url, String contentNamespace, String datasetTitle) {
        BiocaseCrawlConfiguration configuration = new BiocaseCrawlConfiguration(datasetKey, attempt, url, contentNamespace, datasetTitle);
        BiocaseScientificNameRangeRequestHandler requestHandler = new BiocaseScientificNameRangeRequestHandler(configuration);
        BiocaseResponseHandler responseHandler = new BiocaseResponseHandler();
        return Crawler.newCrawler(requestHandler, responseHandler);
    }

    public static Crawler<ScientificNameRangeCrawlContext, String, HttpResponse, List<Byte>> newDigirCrawler(UUID datasetKey, int attempt, URI url, String resourceCode, boolean manis) {
        DigirCrawlConfiguration configuration = new DigirCrawlConfiguration(datasetKey, attempt, url, resourceCode, manis);
        DigirScientificNameRangeRequestHandler requestHandler = new DigirScientificNameRangeRequestHandler(configuration);
        DigirResponseHandler responseHandler = new DigirResponseHandler();
        return Crawler.newCrawler(requestHandler, responseHandler);
    }

    public static <CTX extends CrawlContext, REQ, RESP, RES> Crawler<CTX, REQ, RESP, RES> newInstance(CrawlStrategy<CTX> strategy, RequestHandler<CTX, REQ> requestHandler, ResponseHandler<RESP, RES> responseHandler, CrawlClient<REQ, RESP> client, RetryPolicy retryPolicy, Lock lock) {
        return new Crawler<CTX, REQ, RESP, RES>(strategy, requestHandler, responseHandler, client, retryPolicy, lock);
    }

    public static Crawler<ScientificNameRangeCrawlContext, String, HttpResponse, List<Byte>> newTapirCrawler(UUID datasetKey, int attempt, URI url, String contentNamespace) {
        TapirCrawlConfiguration configuration = new TapirCrawlConfiguration(datasetKey, attempt, url, contentNamespace);
        TapirScientificNameRangeRequestHandler requestHandler = new TapirScientificNameRangeRequestHandler(configuration);
        TapirResponseHandler responseHandler = new TapirResponseHandler();
        return Crawler.newCrawler(requestHandler, responseHandler);
    }

    private static Crawler<ScientificNameRangeCrawlContext, String, HttpResponse, List<Byte>> newCrawler(RequestHandler<ScientificNameRangeCrawlContext, String> requestHandler, ResponseHandler<HttpResponse, List<Byte>> responseHandler) {
        ScientificNameRangeCrawlContext context = new ScientificNameRangeCrawlContext();
        ScientificNameRangeStrategy strategy = new ScientificNameRangeStrategy(context);
        HttpCrawlClient crawlClient = HttpCrawlClient.newInstance(600000, 500, 20);
        LimitedRetryPolicy retryPolicy = new LimitedRetryPolicy(10, 2, 10, 2);
        Lock lock = NoLockFactory.getLock();
        return Crawler.newInstance(strategy, requestHandler, responseHandler, crawlClient, retryPolicy, lock);
    }

    public void addListener(CrawlListener<CTX, REQ, RES> listener) {
        this.listeners.add(listener);
    }

    public void crawl() {
        this.notifyStart();
        Object currentContext = null;
        try {
            while (!(currentContext != null && ((CrawlContext)currentContext).isAborted() || this.retryPolicy.abortCrawl() || !this.strategy.hasNext())) {
                boolean hasNextPage;
                currentContext = this.strategy.next();
                do {
                    this.notifyProgress(currentContext);
                    REQ req = this.requestHandler.buildRequestUrl(currentContext);
                    this.executeRequest(req, this.lock);
                    hasNextPage = this.setupNextCrawlContext(currentContext);
                } while (!((CrawlContext)currentContext).isAborted() && !this.retryPolicy.abortCrawl() && hasNextPage);
            }
        }
        catch (FatalCrawlException e) {
            this.notifyError(e);
            this.notifyFinishedAbnormally();
            return;
        }
        if (currentContext != null && ((CrawlContext)currentContext).isAborted()) {
            LOG.info("Aborted crawl on request");
            this.notifyFinishedOnUserRequest();
        } else if (this.retryPolicy.abortCrawl()) {
            LOG.info("Aborted crawl due to exhausted retries");
            this.notifyFinishedAbnormally();
        } else {
            this.notifyFinishedNormally();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RES executeRequest(REQ req, Lock lock) throws FatalCrawlException {
        boolean complete = false;
        boolean tryAgain = true;
        int tryCount = 0;
        RES res = null;
        do {
            try {
                ++tryCount;
                while (!lock.tryLock(10L, TimeUnit.MINUTES)) {
                    LOG.debug("Failed to acquire lock [{}]", (Object)lock);
                }
                this.notifyRequest(req, tryCount);
                this.stopwatch.reset();
                this.stopwatch.start();
                res = this.client.execute(req, this.responseHandler);
                this.stopwatch.stop();
                this.notifyResponse(res, tryCount, this.stopwatch.elapsed(TimeUnit.MILLISECONDS), this.responseHandler.getRecordCount(), this.responseHandler.isEndOfRecords());
                this.retryPolicy.successfulRequest();
                complete = true;
            }
            catch (TransportException e) {
                this.notifyError(e);
                if (this.retryPolicy.allowAfterTransportException()) {
                    LOG.info("Got transport exception, will retry", (Throwable)e);
                    tryAgain = true;
                    continue;
                }
                tryAgain = false;
                LOG.warn("Got transport exception, will give up this request");
                this.retryPolicy.giveUpRequest();
            }
            catch (ProtocolException e) {
                this.notifyError(e);
                if (this.retryPolicy.allowAfterProtocolException()) {
                    LOG.info("Got protocol exception, will retry", (Throwable)e);
                    tryAgain = true;
                    continue;
                }
                tryAgain = false;
                LOG.warn("Got protocol exception, will give up this request");
                this.retryPolicy.giveUpRequest();
            }
            finally {
                if (this.stopwatch.isRunning()) {
                    this.stopwatch.stop();
                }
                lock.unlock();
            }
        } while (!complete && tryAgain);
        return res;
    }

    private void nextPage(CTX context, boolean speculative) {
        ((CrawlContext)context).setSpeculative(speculative);
        int offset = (Integer)this.responseHandler.getRecordCount().or((Object)this.requestHandler.getLimit());
        ((CrawlContext)context).setOffset(((CrawlContext)context).getOffset() + offset);
    }

    private void nextRange(CTX context) {
        ((CrawlContext)context).setOffset(0);
        ((CrawlContext)context).setSpeculative(false);
    }

    private void notifyError(String msg) {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.error(msg);
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyError(Throwable e) {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.error(e);
            }
            catch (Exception innerException) {
                LOG.warn("Listener threw exception", (Throwable)innerException);
            }
        }
    }

    private void notifyFinishedAbnormally() {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.finishCrawlAbnormally();
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyFinishedNormally() {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.finishCrawlNormally();
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyFinishedOnUserRequest() {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.finishCrawlOnUserRequest();
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyProgress(CTX context) {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.progress(context);
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyRequest(REQ req, int tryCount) {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.request(req, tryCount);
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyResponse(RES result, int tryCount, long duration, Optional<Integer> recordCount, Optional<Boolean> endOfRecords) {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.response(result, tryCount, duration, recordCount, endOfRecords);
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private void notifyStart() {
        for (CrawlListener<CTX, REQ, RES> listener : this.listeners) {
            try {
                listener.startCrawl();
            }
            catch (Exception e) {
                LOG.warn("Listener threw exception", (Throwable)e);
            }
        }
    }

    private boolean setupNextCrawlContext(CTX context) {
        if (!this.responseHandler.isValidState()) {
            this.nextRange(context);
            return false;
        }
        Optional<Long> currentContentHash = this.responseHandler.getContentHash();
        if (currentContentHash.isPresent() && ((CrawlContext)context).getLastContentHash().isPresent() && ((Long)currentContentHash.get()).equals(((CrawlContext)context).getLastContentHash().get())) {
            this.notifyError("Got same content twice in a row, hash is [" + currentContentHash.get() + "]");
            this.nextRange(context);
            return false;
        }
        if (currentContentHash.isPresent()) {
            ((CrawlContext)context).setLastContentHash(currentContentHash);
        }
        if (!currentContentHash.isPresent() && ((CrawlContext)context).isSpeculative()) {
            this.nextRange(context);
            return false;
        }
        if (!currentContentHash.isPresent() || ((CrawlContext)context).isSpeculative()) {
            // empty if block
        }
        Optional<Integer> recordCount = this.responseHandler.getRecordCount();
        if (((Boolean)this.responseHandler.isEndOfRecords().or((Object)false)).booleanValue()) {
            this.nextRange(context);
            if (recordCount.isPresent()) {
                if (currentContentHash.isPresent()) {
                    if ((Integer)recordCount.get() == 0) {
                        this.notifyError("Got content but record count says we got nothing");
                    }
                } else if ((Integer)recordCount.get() != 0) {
                    this.notifyError("Did not get content but record count says we got [" + recordCount.get() + "]" + " records");
                }
            }
            return false;
        }
        if (recordCount.isPresent()) {
            if (currentContentHash.isPresent()) {
                if ((Integer)recordCount.get() == 0) {
                    this.notifyError("Got content but record count says we got nothing");
                }
                this.nextPage(context, false);
                return true;
            }
            if ((Integer)recordCount.get() != 0) {
                this.notifyError("Did not get content but record count says we got [" + recordCount.get() + "] records");
            }
            this.nextPage(context, true);
            return true;
        }
        if (currentContentHash.isPresent()) {
            this.nextPage(context, false);
            return true;
        }
        this.nextPage(context, true);
        return true;
    }
}

