/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.common.parsers.geospatial;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.commons.math3.util.Precision;
import org.gbif.common.parsers.geospatial.LatLngBoundingBox;
import org.gbif.common.parsers.geospatial.UnableToGenerateCellIdException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CellIdUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(CellIdUtils.class);
    private static final int MAX_LATITUDE = 90;
    private static final int MIN_LATITUDE = -90;
    private static final int MAX_LONGITUDE = 180;
    private static final int MIN_LONGITUDE = -180;

    private CellIdUtils() {
        throw new UnsupportedOperationException("Can't initialize class");
    }

    public static int toCellId(Double latitude, Double longitude) throws UnableToGenerateCellIdException {
        LOGGER.debug("Getting cell for [{},{}]", (Object)latitude, (Object)longitude);
        if (latitude == null || latitude < -90.0 || latitude > 90.0 || longitude < -180.0 || longitude > 180.0) {
            throw new UnableToGenerateCellIdException("Latitude[" + latitude + "], Longitude[" + longitude + "] cannot be converted to a cell id");
        }
        int la = CellIdUtils.getCellIdFor(latitude);
        int lo = CellIdUtils.getMod360CellIdFor(longitude);
        return la + lo;
    }

    public static int getMod360CellIdFor(double longitude) {
        return new Double(Math.floor(longitude + 180.0)).intValue();
    }

    public static int getCellIdFor(double latitude) {
        return new Double(Math.floor(latitude + 90.0)).intValue() * 360;
    }

    public static boolean isCellIdInBoundingBox(int cellId, int minCellId, int maxCellId) throws Exception {
        return cellId >= minCellId && cellId <= maxCellId - 361 && cellId % 360 >= minCellId % 360 && cellId % 360 <= (maxCellId - 361) % 360;
    }

    public static int toCentiCellId(Double latitude, Double longitude) throws UnableToGenerateCellIdException {
        if (latitude == null || latitude < -90.0 || latitude > 90.0 || longitude < -180.0 || longitude > 180.0) {
            throw new UnableToGenerateCellIdException("Latitude[" + latitude + "], Longitude[" + longitude + "] cannot be " + "converted to a centi cell id");
        }
        int la = Math.abs((int)(latitude * 10000.0) % 10000);
        if (latitude < 0.0) {
            la = 10000 - la;
        }
        la = la / 1000 % 10;
        int lo = Math.abs((int)(longitude * 10000.0) % 10000);
        if (longitude < 0.0) {
            lo = 10000 - lo;
        }
        lo = lo / 1000 % 10;
        int centiCellId = la * 10 + lo;
        return Math.abs(centiCellId);
    }

    public static LatLngBoundingBox toBoundingBox(int cellId) {
        int longitude = cellId % 360 - 180;
        int latitude = -90;
        if (cellId > 0) {
            latitude = new Double(Math.floor(cellId / 360)).intValue() - 90;
        }
        return new LatLngBoundingBox(longitude, latitude, longitude + 1, latitude + 1);
    }

    public static LatLngBoundingBox toBoundingBox(int cellId, int centiCellId) {
        int longitudeX10 = 10 * (cellId % 360 - 180);
        int latitudeX10 = -900;
        if (cellId > 0) {
            latitudeX10 = 10 * (new Double(Math.floor(cellId / 360)).intValue() - 90);
        }
        double longOffset = centiCellId % 10;
        double latOffset = 0.0;
        if (centiCellId > 0) {
            latOffset = centiCellId / 10;
        }
        double minLatitude = ((double)latitudeX10 + latOffset) / 10.0;
        double minLongitude = ((double)longitudeX10 + longOffset) / 10.0;
        double maxLatitude = ((double)latitudeX10 + latOffset + 1.0) / 10.0;
        double maxlongitude = ((double)longitudeX10 + longOffset + 1.0) / 10.0;
        return new LatLngBoundingBox(minLongitude, minLatitude, maxlongitude, maxLatitude);
    }

    public static Set<Integer> getCellsEnclosedBy(double minLat, double maxLat, double minLong, double maxLong) throws UnableToGenerateCellIdException {
        if (minLat < -90.0) {
            minLat = -90.0;
        }
        if (maxLat > 90.0) {
            maxLat = 90.0;
        }
        if (minLong < -180.0) {
            minLong = -180.0;
        }
        if (maxLong > 180.0) {
            maxLong = 180.0;
        }
        LOGGER.debug("Establishing cells enclosed by: {}:{}   {}:{}", new Object[]{minLat, maxLat, minLong, maxLong});
        int lower = CellIdUtils.toCellId(minLat, minLong);
        int upper = CellIdUtils.toCellId(maxLat, maxLong);
        LOGGER.debug("Unprocessed cells: {} -> {}", (Object)lower, (Object)upper);
        if (Math.ceil(maxLong) == Math.floor(maxLong)) {
            LOGGER.debug("Longitude lies on a boundary");
            --upper;
        }
        if (Math.ceil(maxLat) == Math.floor(maxLat)) {
            LOGGER.debug("Latitude lies on a boundary");
            upper -= 360;
        }
        LOGGER.debug("Getting cells contained in {} to {}", (Object)lower, (Object)upper);
        int omitLeft = lower % 360;
        int omitRight = upper % 360;
        if (omitRight == 0) {
            omitRight = 360;
        }
        HashSet<Integer> cells = new HashSet<Integer>();
        for (int i = lower; i <= upper; ++i) {
            if (i % 360 < omitLeft || i % 360 > omitRight) continue;
            cells.add(i);
        }
        return cells;
    }

    public static int[] getMinMaxCellIdsForBoundingBox(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude) throws UnableToGenerateCellIdException {
        int minCellId = CellIdUtils.toCellId(minLatitude, minLongitude);
        int maxCellId = CellIdUtils.toCellId(maxLatitude, maxLongitude);
        if (Math.ceil(maxLatitude) == Math.floor(maxLatitude) && Math.ceil(maxLongitude) == Math.floor(maxLongitude) && maxLongitude != 180.0 && maxLatitude != 90.0 && maxCellId > 0) {
            maxCellId -= 361;
        }
        return new int[]{minCellId, maxCellId};
    }

    public static LatLngBoundingBox getBoundingBoxForCells(List<Integer> cellIds) {
        if (cellIds.isEmpty()) {
            return null;
        }
        int minLatitudeCellId = cellIds.get(0);
        int maxLatitudeCellId = cellIds.get(cellIds.size() - 1);
        int minLongitudeCellId = cellIds.get(0);
        int maxLongitudeCellId = cellIds.get(cellIds.size() - 1);
        for (Integer cellId : cellIds) {
            Integer cellIdMod360 = cellId % 360;
            if (cellIdMod360 < minLongitudeCellId % 360) {
                minLongitudeCellId = cellIdMod360;
            }
            if (cellIdMod360 > maxLongitudeCellId % 360) {
                maxLongitudeCellId = cellIdMod360;
            }
            if (cellId < minLatitudeCellId) {
                minLatitudeCellId = cellId;
            }
            if (cellId <= maxLatitudeCellId) continue;
            maxLatitudeCellId = cellId;
        }
        double minLongitude = CellIdUtils.toBoundingBox((int)minLongitudeCellId).minLong;
        double minLatitude = CellIdUtils.toBoundingBox((int)minLatitudeCellId).minLat;
        double maxLongitude = CellIdUtils.toBoundingBox((int)maxLongitudeCellId).maxLong;
        double maxLatitude = CellIdUtils.toBoundingBox((int)maxLatitudeCellId).maxLat;
        return new LatLngBoundingBox(minLongitude, minLatitude, maxLongitude, maxLatitude);
    }

    public static Integer[] getCentiCellIdForBoundingBox(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude) throws UnableToGenerateCellIdException {
        if (!CellIdUtils.isBoundingBoxCentiCell(minLongitude, minLatitude, maxLongitude, maxLatitude)) {
            return null;
        }
        int[] maxMinCellIds = CellIdUtils.getMinMaxCellIdsForBoundingBox(minLongitude, minLatitude, maxLongitude, maxLatitude);
        Integer cellid = maxMinCellIds[0];
        int minCentiCell = CellIdUtils.toCentiCellId(minLatitude, minLongitude);
        int maxCentiCell = CellIdUtils.toCentiCellId(maxLatitude, maxLongitude);
        double maxLongitude10 = maxLongitude * 10.0;
        double maxLatitude10 = maxLatitude * 10.0;
        if (Math.ceil(maxLatitude10) == Math.floor(maxLatitude10) && Math.ceil(maxLongitude10) == Math.floor(maxLongitude10) && maxCentiCell > 0) {
            maxCentiCell = maxCentiCell > minCentiCell ? (maxCentiCell -= 11) : (maxCentiCell += 9);
        }
        return new Integer[]{cellid, minCentiCell};
    }

    private static boolean isBoundingBoxCentiCell(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude) {
        double width = maxLongitude > minLongitude ? maxLongitude - minLongitude : minLongitude - maxLongitude;
        double height = maxLatitude > minLatitude ? maxLatitude - minLatitude : minLatitude - maxLatitude;
        return Precision.round((double)height, (int)1) == (double)0.1f && Precision.round((double)width, (int)1) == (double)0.1f;
    }

    public static void main(String[] args) {
        try {
            if (args.length == 1) {
                LatLngBoundingBox llbb = CellIdUtils.toBoundingBox(Integer.parseInt(args[0]));
                System.out.println("CellId " + args[0] + ": minX[" + llbb.getMinLong() + "] minY[" + llbb.getMinLat() + "] maxX[" + llbb.getMaxLong() + "] maxY[" + llbb.getMaxLat() + "]");
            } else if (args.length == 2) {
                double lat = Double.parseDouble(args[0]);
                double lon = Double.parseDouble(args[1]);
                System.out.println("lat[" + lat + "] long[" + lon + "] = cellId: " + CellIdUtils.toCellId(lat, lon));
            } else {
                System.out.println("Provide either a 'cell id' or 'Lat Long' params!");
            }
        }
        catch (NumberFormatException e) {
            e.printStackTrace();
        }
        catch (UnableToGenerateCellIdException e) {
            e.printStackTrace();
        }
    }
}

