/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.dwca.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOCase;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.commons.io.input.BOMInputStream;
import org.gbif.dwc.terms.DcTerm;
import org.gbif.dwc.terms.DwcTerm;
import org.gbif.dwc.terms.Term;
import org.gbif.dwc.terms.TermFactory;
import org.gbif.dwca.io.Archive;
import org.gbif.dwca.io.ArchiveField;
import org.gbif.dwca.io.ArchiveFile;
import org.gbif.dwca.io.MetaXMLSaxHandler;
import org.gbif.dwca.io.UnsupportedArchiveException;
import org.gbif.util.DownloadUtil;
import org.gbif.utils.file.CompressionUtil;
import org.gbif.utils.file.csv.CSVReader;
import org.gbif.utils.file.csv.CSVReaderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.helpers.DefaultHandler;

public class ArchiveFactory {
    private static final TermFactory TERM_FACTORY = TermFactory.instance();
    private static final Logger LOG = LoggerFactory.getLogger(ArchiveFactory.class);
    private static final List<String> DATA_FILE_SUFFICES = ImmutableList.of((Object)".csv", (Object)".txt", (Object)".tsv", (Object)".tab", (Object)".text", (Object)".data", (Object)".dwca");
    private static final Map<Term, Term> TERM_TO_ROW_TYPE;
    private static final List<Term> ID_TERMS;
    private static final SAXParserFactory SAX_FACTORY;

    public static Archive openArchive(URL archiveUrl, File workingDir) throws IOException, UnsupportedArchiveException {
        File downloadTo = new File(workingDir, "dwca-download");
        File dwca = new File(workingDir, "dwca");
        DownloadUtil.download(archiveUrl, downloadTo);
        return ArchiveFactory.openArchive(downloadTo, dwca);
    }

    public static Archive openArchive(File archiveFile, File archiveDir) throws IOException, UnsupportedArchiveException {
        if (archiveDir.exists()) {
            LOG.debug("Deleting existing archive folder [{}]", (Object)archiveDir.getAbsolutePath());
            org.gbif.utils.file.FileUtils.deleteDirectoryRecursively((File)archiveDir);
        }
        FileUtils.forceMkdir((File)archiveDir);
        try {
            File root;
            CompressionUtil.decompressFile((File)archiveDir, (File)archiveFile, (boolean)true);
            File[] rootFiles = archiveDir.listFiles((FileFilter)HiddenFileFilter.VISIBLE);
            if (rootFiles.length == 1 && (root = rootFiles[0]).isDirectory()) {
                LOG.debug("Removing single root folder {} found in decompressed archive", (Object)root.getAbsoluteFile());
                for (File f : FileUtils.listFiles((File)root, (IOFileFilter)TrueFileFilter.TRUE, null)) {
                    File f2 = new File(archiveDir, f.getName());
                    f.renameTo(f2);
                }
            }
            return ArchiveFactory.openArchive(archiveDir);
        }
        catch (CompressionUtil.UnsupportedCompressionType e) {
            LOG.debug("Could not uncompress archive [{}], try to read as single text file", (Object)archiveFile, (Object)e);
            Archive arch = ArchiveFactory.openArchiveDataFile(archiveFile);
            Files.copy((File)archiveFile, (File)new File(archiveDir, archiveFile.getName()));
            return arch;
        }
    }

    public static Archive openArchiveDataFile(File dataFile) throws IOException, UnsupportedArchiveException {
        Archive archive = new Archive();
        archive.setLocation(dataFile);
        ArchiveFile coreFile = ArchiveFactory.readFileHeaders(dataFile);
        archive.setCore(coreFile);
        ArchiveFactory.discoverMetadataFile(archive, dataFile.getParentFile());
        return ArchiveFactory.validateArchive(archive);
    }

    public static Archive openArchive(File dwcaFolder) throws IOException, UnsupportedArchiveException {
        if (!dwcaFolder.exists()) {
            throw new FileNotFoundException("Archive folder not existing: " + dwcaFolder.getAbsolutePath());
        }
        if (dwcaFolder.isFile()) {
            return ArchiveFactory.openArchiveDataFile(dwcaFolder);
        }
        Archive archive = new Archive();
        archive.setLocation(dwcaFolder);
        Iterator iter = FileUtils.iterateFiles((File)dwcaFolder, (String[])new String[]{"xml", "txt"}, (boolean)false);
        while (iter.hasNext()) {
            File f = (File)iter.next();
            if (!f.getName().startsWith("\\")) continue;
            String orig = f.getName();
            String replacement = f.getName().replaceFirst("\\\\", "");
            LOG.info("Renaming file from {} to {}", (Object)orig, (Object)replacement);
            f.renameTo(new File(dwcaFolder, replacement));
        }
        File mf = new File(dwcaFolder, "meta.xml");
        if (mf.exists()) {
            ArchiveFactory.readMetaDescriptor(archive, new FileInputStream(mf));
        } else {
            ArrayList<File> dataFiles = new ArrayList<File>();
            for (String suffix : DATA_FILE_SUFFICES) {
                IOFileFilter ff = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)suffix, (IOCase)IOCase.INSENSITIVE), HiddenFileFilter.VISIBLE});
                dataFiles.addAll(Arrays.asList(dwcaFolder.listFiles((FileFilter)ff)));
            }
            if (dataFiles.size() == 1) {
                File dataFile = new File(dwcaFolder, ((File)dataFiles.get(0)).getName());
                ArchiveFile coreFile = ArchiveFactory.readFileHeaders(dataFile);
                coreFile.getLocations().clear();
                coreFile.addLocation(dataFile.getName());
                archive.setCore(coreFile);
            } else {
                throw new UnsupportedArchiveException("The archive given is a folder with more or less than 1 data files having a csv, txt or tab suffix");
            }
        }
        ArchiveFactory.discoverMetadataFile(archive, mf.getParentFile());
        return ArchiveFactory.validateArchive(archive);
    }

    private static void discoverMetadataFile(Archive archive, File folder) {
        if (archive.getMetadataLocation() == null) {
            for (String metadataFN : Lists.newArrayList((Object[])new String[]{"eml.xml", "metadata.xml"})) {
                File emlFile = new File(folder, metadataFN);
                if (!emlFile.exists()) continue;
                archive.setMetadataLocation(metadataFN);
                break;
            }
        }
    }

    private static ArchiveFile readFileHeaders(File dataFile) throws UnsupportedArchiveException, IOException {
        String[] headers;
        ArchiveFile dwcFile = new ArchiveFile();
        dwcFile.addLocation(null);
        dwcFile.setIgnoreHeaderLines(1);
        String[] stringArray = null;
        try (CSVReader reader = CSVReaderFactory.build((File)dataFile);){
            dwcFile.setEncoding(reader.encoding);
            dwcFile.setFieldsTerminatedBy(reader.delimiter);
            dwcFile.setFieldsEnclosedBy(reader.quoteChar);
            headers = reader.getHeader();
        }
        catch (Throwable object) {
            stringArray = object;
            throw object;
        }
        int index = 0;
        for (String head : headers) {
            if (head != null && head.length() > 1) {
                try {
                    Term dt = TERM_FACTORY.findTerm(head);
                    ArchiveField field = new ArchiveField(index, dt, null, ArchiveField.DataType.string);
                    dwcFile.addField(field);
                }
                catch (IllegalArgumentException e) {
                    LOG.warn("Illegal term name >>{}<< found in header, ignore column {}", (Object)head, (Object)index);
                }
            }
            ++index;
        }
        List<Term> headerAsTerm = dwcFile.getFields().keySet().stream().collect(Collectors.toList());
        ArchiveFactory.determineRecordIdentifier(headerAsTerm).ifPresent(t -> dwcFile.setId(dwcFile.getField((Term)t)));
        ArchiveFactory.determineRowType(headerAsTerm).ifPresent(dwcFile::setRowType);
        return dwcFile;
    }

    @VisibleForTesting
    protected static void readMetaDescriptor(Archive archive, InputStream metaDescriptor) throws UnsupportedArchiveException {
        try (BOMInputStream bomInputStream = new BOMInputStream(metaDescriptor);){
            SAXParser p = SAX_FACTORY.newSAXParser();
            MetaXMLSaxHandler mh = new MetaXMLSaxHandler(archive);
            LOG.debug("Reading archive metadata file");
            p.parse((InputStream)bomInputStream, (DefaultHandler)mh);
        }
        catch (Exception e1) {
            LOG.warn("Exception caught", (Throwable)e1);
            throw new UnsupportedArchiveException(e1);
        }
    }

    private static Archive validateArchive(Archive archive) throws UnsupportedArchiveException {
        ArchiveFactory.validateCoreFile(archive.getCore(), !archive.getExtensions().isEmpty());
        for (ArchiveFile af : archive.getExtensions()) {
            ArchiveFactory.validateExtensionFile(af);
        }
        LOG.debug("Archive contains " + archive.getExtensions().size() + " described extension files");
        LOG.debug("Archive contains " + archive.getCore().getFields().size() + " core properties");
        return archive;
    }

    private static void validateCoreFile(ArchiveFile f, boolean hasExtensions) throws UnsupportedArchiveException {
        if (hasExtensions && f.getId() == null) {
            LOG.warn("DwC-A core data file " + f.getTitle() + " is lacking an id column. No extensions allowed in this case");
        }
        ArchiveFactory.validateFile(f);
    }

    private static void validateExtensionFile(ArchiveFile f) throws UnsupportedArchiveException {
        if (f.getId() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires an id or foreign key to the core id");
        }
        ArchiveFactory.validateFile(f);
    }

    private static void validateFile(ArchiveFile f) throws UnsupportedArchiveException {
        if (f == null) {
            throw new UnsupportedArchiveException("DwC-A data file is NULL");
        }
        if (f.getLocationFile() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires a location");
        }
        if (f.getEncoding() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires a character encoding");
        }
    }

    static Optional<Term> determineRowType(List<Term> terms) {
        return TERM_TO_ROW_TYPE.entrySet().stream().filter(ke -> terms.contains(ke.getKey())).map(Map.Entry::getValue).findFirst();
    }

    static Optional<Term> determineRecordIdentifier(List<Term> terms) {
        return ID_TERMS.stream().filter(terms::contains).findFirst();
    }

    static {
        LinkedHashMap<DwcTerm, DwcTerm> idToRowType = new LinkedHashMap<DwcTerm, DwcTerm>();
        idToRowType.put(DwcTerm.occurrenceID, DwcTerm.Occurrence);
        idToRowType.put(DwcTerm.taxonID, DwcTerm.Taxon);
        idToRowType.put(DwcTerm.eventID, DwcTerm.Event);
        TERM_TO_ROW_TYPE = Collections.unmodifiableMap(idToRowType);
        ID_TERMS = Collections.unmodifiableList(Arrays.asList(DwcTerm.occurrenceID, DwcTerm.taxonID, DwcTerm.eventID, DcTerm.identifier));
        SAX_FACTORY = SAXParserFactory.newInstance();
        SAX_FACTORY.setNamespaceAware(true);
        SAX_FACTORY.setValidating(false);
    }
}

