/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.common.parsers.geospatial;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import java.util.HashSet;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.gbif.api.vocabulary.OccurrenceIssue;
import org.gbif.common.parsers.NumberParser;
import org.gbif.common.parsers.core.OccurrenceParseResult;
import org.gbif.common.parsers.core.ParseResult;
import org.gbif.common.parsers.geospatial.DoubleAccuracy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MeterRangeParser {
    private static final Logger LOG = LoggerFactory.getLogger(MeterRangeParser.class);
    private static final Pattern MEASURE_MARKER_PATTERN = Pattern.compile(".*[a-zA-Z].*");
    private static final Pattern REMOVE_MEASURE_MARKER_PATTERN = Pattern.compile("[a-zA-Z\" \"\"]");
    private static final Pattern FEET_MARKER_PATTERN = Pattern.compile(".*ft.*|.*'.*", 2);
    private static final Pattern INCHES_MARKER_PATTERN = Pattern.compile(".*in.*|.*\".*", 2);
    private static final Pattern SEP_MARKER_PATTERN = Pattern.compile("\\d-.*");
    private static final float FEET_TO_METRES = 0.3048f;
    private static final float INCHES_TO_METRES = 0.0254f;
    private static final int MIN_ELEVATION = -11000;
    private static final int MAX_ELEVATION = 17000;
    private static final int MAX_DEPTH = Math.abs(-11000);
    private static final int MIN_DISTANCE = -5000;
    private static final int MAX_DISTANCE = 17000;

    public static MeasurementWrapper<DoubleAccuracy> parseMeterRange(String minRaw, @Nullable String maxRaw, @Nullable String precisionRaw) {
        Double accuracy;
        Double value;
        MeasurementWrapper<DoubleAccuracy> result = new MeasurementWrapper<DoubleAccuracy>();
        MeasurementWrapper<Double> min = MeterRangeParser.parseInMeter(minRaw);
        MeasurementWrapper<Double> max = MeterRangeParser.parseInMeter(maxRaw);
        MeasurementWrapper<Double> prec = MeterRangeParser.parseInMeter(precisionRaw);
        result.addIssues(min);
        result.addIssues(max);
        result.addIssues(prec);
        if (((MeasurementWrapper)min).measurement == null && ((MeasurementWrapper)max).measurement == null) {
            return result;
        }
        if (((MeasurementWrapper)min).measurement != null && ((MeasurementWrapper)max).measurement != null) {
            if ((Double)((MeasurementWrapper)min).measurement > (Double)((MeasurementWrapper)max).measurement) {
                ((MeasurementWrapper)result).minMaxSwapped = true;
                Double oldMin = (Double)((MeasurementWrapper)min).measurement;
                ((MeasurementWrapper)min).measurement = ((MeasurementWrapper)max).measurement;
                ((MeasurementWrapper)max).measurement = oldMin;
            }
            if (((MeasurementWrapper)prec).measurement != null) {
                MeasurementWrapper<Double> measurementWrapper = min;
                ((MeasurementWrapper)measurementWrapper).measurement = (Double)((MeasurementWrapper)measurementWrapper).measurement - (Double)((MeasurementWrapper)prec).measurement;
                measurementWrapper = max;
                ((MeasurementWrapper)measurementWrapper).measurement = (Double)((MeasurementWrapper)measurementWrapper).measurement + (Double)((MeasurementWrapper)prec).measurement;
            }
            value = ((Double)((MeasurementWrapper)min).measurement + (Double)((MeasurementWrapper)max).measurement) / 2.0;
            accuracy = ((Double)((MeasurementWrapper)max).measurement - (Double)((MeasurementWrapper)min).measurement) / 2.0;
        } else {
            value = ((MeasurementWrapper)min).measurement == null ? (Double)((MeasurementWrapper)max).measurement : (Double)((MeasurementWrapper)min).measurement;
            accuracy = (Double)((MeasurementWrapper)prec).measurement;
        }
        if (value != null) {
            ((MeasurementWrapper)result).measurement = new DoubleAccuracy(value, accuracy);
        }
        return result;
    }

    public static OccurrenceParseResult<DoubleAccuracy> parseElevation(@Nullable String min, @Nullable String max, @Nullable String precision) {
        MeasurementWrapper<DoubleAccuracy> elevation = MeterRangeParser.parseMeterRange(min, max, precision);
        HashSet issues = Sets.newHashSet();
        if (((MeasurementWrapper)elevation).containsNonNumeric) {
            issues.add(OccurrenceIssue.ELEVATION_NON_NUMERIC);
        }
        if (((MeasurementWrapper)elevation).isInFeet || ((MeasurementWrapper)elevation).isInInches) {
            issues.add(OccurrenceIssue.ELEVATION_NOT_METRIC);
        }
        if (((MeasurementWrapper)elevation).minMaxSwapped) {
            issues.add(OccurrenceIssue.ELEVATION_MIN_MAX_SWAPPED);
        }
        if (((MeasurementWrapper)elevation).tooLarge) {
            issues.add(OccurrenceIssue.ELEVATION_UNLIKELY);
        }
        if (((MeasurementWrapper)elevation).measurement == null || ((DoubleAccuracy)((MeasurementWrapper)elevation).measurement).getValue() == null) {
            return OccurrenceParseResult.fail(issues);
        }
        DoubleAccuracy result = (DoubleAccuracy)((MeasurementWrapper)elevation).measurement;
        if (result.getValue() > 17000.0 || result.getValue() < -11000.0) {
            issues.add(OccurrenceIssue.ELEVATION_UNLIKELY);
            return OccurrenceParseResult.fail(issues);
        }
        return OccurrenceParseResult.success(ParseResult.CONFIDENCE.DEFINITE, result, issues);
    }

    public static OccurrenceParseResult<DoubleAccuracy> parseDepth(@Nullable String min, @Nullable String max, @Nullable String precision) {
        MeasurementWrapper<DoubleAccuracy> depth = MeterRangeParser.parseMeterRange(min, max, precision);
        HashSet issues = Sets.newHashSet();
        if (((MeasurementWrapper)depth).containsNonNumeric) {
            issues.add(OccurrenceIssue.DEPTH_NON_NUMERIC);
        }
        if (((MeasurementWrapper)depth).isInFeet || ((MeasurementWrapper)depth).isInInches) {
            issues.add(OccurrenceIssue.DEPTH_NOT_METRIC);
        }
        if (((MeasurementWrapper)depth).minMaxSwapped) {
            issues.add(OccurrenceIssue.DEPTH_MIN_MAX_SWAPPED);
        }
        if (((MeasurementWrapper)depth).tooLarge) {
            issues.add(OccurrenceIssue.DEPTH_UNLIKELY);
        }
        if (((MeasurementWrapper)depth).measurement == null || ((DoubleAccuracy)((MeasurementWrapper)depth).measurement).getValue() == null) {
            return OccurrenceParseResult.fail(issues);
        }
        DoubleAccuracy result = (DoubleAccuracy)((MeasurementWrapper)depth).measurement;
        if (result.getValue() < 0.0) {
            result = new DoubleAccuracy(-1.0 * result.getValue(), result.getAccuracy());
            issues.add(OccurrenceIssue.DEPTH_UNLIKELY);
        }
        if (result.getValue() > (double)MAX_DEPTH) {
            issues.add(OccurrenceIssue.DEPTH_UNLIKELY);
            return OccurrenceParseResult.fail(issues);
        }
        return OccurrenceParseResult.success(ParseResult.CONFIDENCE.DEFINITE, result, issues);
    }

    public static ParseResult<Double> parseMeters(String meter) {
        MeasurementWrapper<Double> result = MeterRangeParser.parseInMeter(meter);
        if (result.getMeasurement() == null) {
            return ParseResult.fail();
        }
        return ParseResult.success(ParseResult.CONFIDENCE.DEFINITE, result.getMeasurement());
    }

    private static MeasurementWrapper<Double> parseInMeter(String meter) {
        MeasurementWrapper<Double> iMeter;
        block14: {
            iMeter = new MeasurementWrapper<Double>();
            if (Strings.isNullOrEmpty((String)meter)) {
                return iMeter;
            }
            try {
                ((MeasurementWrapper)iMeter).containsNonNumeric = MeterRangeParser.MEASURE_MARKER_PATTERN.matcher(meter).matches();
                if (!iMeter.containsNonNumeric()) {
                    ((MeasurementWrapper)iMeter).measurement = NumberParser.parseDouble(meter);
                    break block14;
                }
                ((MeasurementWrapper)iMeter).isInFeet = MeterRangeParser.FEET_MARKER_PATTERN.matcher(meter).matches();
                ((MeasurementWrapper)iMeter).isInInches = MeterRangeParser.INCHES_MARKER_PATTERN.matcher(meter).matches();
                if (SEP_MARKER_PATTERN.matcher(meter).matches()) {
                    try {
                        String min = meter.substring(0, meter.indexOf(45)).trim();
                        min = MeterRangeParser.removeMeasurementMarkers(min);
                        String max = meter.substring(meter.indexOf(45) + 1).trim();
                        max = MeterRangeParser.removeMeasurementMarkers(max);
                        Double minDouble = NumberParser.parseDouble(min);
                        Double maxDouble = NumberParser.parseDouble(max);
                        if (minDouble != 0.0 && maxDouble != 0.0 && maxDouble - minDouble != 0.0) {
                            ((MeasurementWrapper)iMeter).measurement = (maxDouble + minDouble) / 2.0;
                        }
                    }
                    catch (NumberFormatException min) {}
                } else {
                    ((MeasurementWrapper)iMeter).measurement = NumberParser.parseDouble(MeterRangeParser.removeMeasurementMarkers(meter));
                }
                if (((MeasurementWrapper)iMeter).measurement != null) {
                    if (((MeasurementWrapper)iMeter).isInFeet) {
                        ((MeasurementWrapper)iMeter).measurement = MeterRangeParser.convertFeetToMetres((Double)((MeasurementWrapper)iMeter).measurement);
                    } else if (((MeasurementWrapper)iMeter).isInInches) {
                        ((MeasurementWrapper)iMeter).measurement = MeterRangeParser.convertInchesToMetres((Double)((MeasurementWrapper)iMeter).measurement);
                    }
                }
            }
            catch (NumberFormatException e) {
                LOG.debug("Unparsable meter measurement: {}", (Object)meter, (Object)e.getMessage());
            }
        }
        if (((MeasurementWrapper)iMeter).measurement != null) {
            ((MeasurementWrapper)iMeter).measurement = (double)Math.round((Double)((MeasurementWrapper)iMeter).measurement * 100.0) / 100.0;
        }
        return iMeter;
    }

    private static String removeMeasurementMarkers(String s) {
        if (s == null) {
            return null;
        }
        return REMOVE_MEASURE_MARKER_PATTERN.matcher(s).replaceAll("");
    }

    private static double convertInchesToMetres(double inches) {
        return inches * (double)0.0254f;
    }

    private static double convertFeetToMetres(double feet) {
        return feet * (double)0.3048f;
    }

    private static Integer roundedInt(Double x) {
        if (x == null) {
            return null;
        }
        Long xl = Math.round(x);
        if (xl > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Long too big for an integer");
        }
        return xl.intValue();
    }

    static class MeasurementWrapper<T> {
        private T measurement;
        private boolean isInFeet;
        private boolean isInInches;
        private boolean containsNonNumeric;
        private boolean minMaxSwapped;
        private boolean tooLarge;

        MeasurementWrapper() {
        }

        public T getMeasurement() {
            return this.measurement;
        }

        public boolean isInFeet() {
            return this.isInFeet;
        }

        public boolean isInInches() {
            return this.isInInches;
        }

        public boolean containsNonNumeric() {
            return this.containsNonNumeric;
        }

        public boolean isMinMaxSwapped() {
            return this.minMaxSwapped;
        }

        public boolean isTooLarge() {
            return this.tooLarge;
        }

        public void addIssues(MeasurementWrapper<?> issues) {
            this.isInFeet = this.isInFeet || issues.isInFeet;
            this.isInInches = this.isInInches || issues.isInInches;
            this.containsNonNumeric = this.containsNonNumeric || issues.containsNonNumeric;
            this.minMaxSwapped = this.minMaxSwapped || issues.minMaxSwapped;
            this.tooLarge = this.tooLarge || issues.tooLarge;
        }
    }
}

