/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.dwca.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOCase;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.commons.io.input.BOMInputStream;
import org.gbif.dwc.terms.DcTerm;
import org.gbif.dwc.terms.DwcTerm;
import org.gbif.dwc.terms.Term;
import org.gbif.dwc.terms.TermFactory;
import org.gbif.dwca.io.Archive;
import org.gbif.dwca.io.ArchiveField;
import org.gbif.dwca.io.ArchiveFile;
import org.gbif.dwca.io.SimpleSaxHandler;
import org.gbif.dwca.io.UnsupportedArchiveException;
import org.gbif.io.DownloadUtil;
import org.gbif.utils.file.CompressionUtil;
import org.gbif.utils.file.csv.CSVReader;
import org.gbif.utils.file.csv.CSVReaderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class ArchiveFactory {
    private static final TermFactory TERM_FACTORY = TermFactory.instance();
    private static final Logger LOG = LoggerFactory.getLogger(ArchiveFactory.class);
    private static final List<String> DATA_FILE_SUFFICES = ImmutableList.of((Object)".csv", (Object)".txt", (Object)".tsv", (Object)".tab", (Object)".text", (Object)".data", (Object)".dwca");
    private static final SAXParserFactory SAX_FACTORY = SAXParserFactory.newInstance();

    public static Archive openArchive(URL archiveUrl, File workingDir) throws IOException, UnsupportedArchiveException {
        File downloadTo = new File(workingDir, "dwca-download");
        File dwca = new File(workingDir, "dwca");
        DownloadUtil.download(archiveUrl, downloadTo);
        return ArchiveFactory.openArchive(downloadTo, dwca);
    }

    public static Archive openArchive(File archiveFile, File archiveDir) throws IOException, UnsupportedArchiveException {
        if (archiveDir.exists()) {
            LOG.debug("Deleting existing archive folder [{}]", (Object)archiveDir.getAbsolutePath());
            org.gbif.utils.file.FileUtils.deleteDirectoryRecursively((File)archiveDir);
        }
        FileUtils.forceMkdir((File)archiveDir);
        try {
            File root;
            CompressionUtil.decompressFile((File)archiveDir, (File)archiveFile, (boolean)true);
            File[] rootFiles = archiveDir.listFiles((FileFilter)HiddenFileFilter.VISIBLE);
            if (rootFiles.length == 1 && (root = rootFiles[0]).isDirectory()) {
                LOG.debug("Removing single root folder {} found in decompressed archive", (Object)root.getAbsoluteFile());
                for (File f : FileUtils.listFiles((File)root, (IOFileFilter)TrueFileFilter.TRUE, null)) {
                    File f2 = new File(archiveDir, f.getName());
                    f.renameTo(f2);
                }
            }
            return ArchiveFactory.openArchive(archiveDir);
        }
        catch (CompressionUtil.UnsupportedCompressionType e) {
            LOG.debug("Could not uncompress archive [{}], try to read as single text file", (Object)archiveFile, (Object)e);
            Archive arch = ArchiveFactory.openArchiveDataFile(archiveFile);
            Files.copy((File)archiveFile, (File)new File(archiveDir, archiveFile.getName()));
            return arch;
        }
    }

    public static Archive openArchiveDataFile(File dataFile) throws IOException, UnsupportedArchiveException {
        Archive archive = new Archive();
        archive.setLocation(dataFile);
        ArchiveFile coreFile = ArchiveFactory.readFileHeaders(dataFile);
        archive.setCore(coreFile);
        ArchiveFactory.discoverMetadataFile(archive, dataFile.getParentFile());
        return ArchiveFactory.validateArchive(archive);
    }

    public static Archive openArchive(File dwcaFolder) throws IOException, UnsupportedArchiveException {
        if (!dwcaFolder.exists()) {
            throw new FileNotFoundException("Archive folder not existing: " + dwcaFolder.getAbsolutePath());
        }
        if (dwcaFolder.isFile()) {
            return ArchiveFactory.openArchiveDataFile(dwcaFolder);
        }
        Archive archive = new Archive();
        archive.setLocation(dwcaFolder);
        Iterator iter = FileUtils.iterateFiles((File)dwcaFolder, (String[])new String[]{"xml", "txt"}, (boolean)false);
        while (iter.hasNext()) {
            File f = (File)iter.next();
            if (!f.getName().startsWith("\\")) continue;
            String orig = f.getName();
            String replacement = f.getName().replaceFirst("\\\\", "");
            LOG.info("Renaming file from {} to {}", (Object)orig, (Object)replacement);
            f.renameTo(new File(dwcaFolder, replacement));
        }
        File mf = new File(dwcaFolder, "meta.xml");
        if (mf.exists()) {
            ArchiveFactory.readMetaDescriptor(archive, new FileInputStream(mf));
        } else {
            ArrayList<File> dataFiles = new ArrayList<File>();
            for (String suffix : DATA_FILE_SUFFICES) {
                IOFileFilter ff = FileFilterUtils.and((IOFileFilter[])new IOFileFilter[]{FileFilterUtils.suffixFileFilter((String)suffix, (IOCase)IOCase.INSENSITIVE), HiddenFileFilter.VISIBLE});
                dataFiles.addAll(Arrays.asList(dwcaFolder.listFiles((FileFilter)ff)));
            }
            if (dataFiles.size() == 1) {
                File dataFile = new File(dwcaFolder, ((File)dataFiles.get(0)).getName());
                ArchiveFile coreFile = ArchiveFactory.readFileHeaders(dataFile);
                coreFile.getLocations().clear();
                coreFile.addLocation(dataFile.getName());
                archive.setCore(coreFile);
            } else {
                throw new UnsupportedArchiveException("The archive given is a folder with more or less than 1 data files having a csv, txt or tab suffix");
            }
        }
        ArchiveFactory.discoverMetadataFile(archive, mf.getParentFile());
        return ArchiveFactory.validateArchive(archive);
    }

    private static void discoverMetadataFile(Archive archive, File folder) {
        if (archive.getMetadataLocation() == null) {
            for (String metadataFN : Lists.newArrayList((Object[])new String[]{"eml.xml", "metadata.xml"})) {
                File emlFile = new File(folder, metadataFN);
                if (!emlFile.exists()) continue;
                archive.setMetadataLocation(metadataFN);
                break;
            }
        }
    }

    private static ArchiveFile readFileHeaders(File dataFile) throws UnsupportedArchiveException, IOException {
        ArchiveFile dwcFile = new ArchiveFile();
        dwcFile.addLocation(null);
        dwcFile.setIgnoreHeaderLines(1);
        CSVReader reader = CSVReaderFactory.build((File)dataFile);
        dwcFile.setEncoding(reader.encoding);
        dwcFile.setFieldsTerminatedBy(reader.delimiter);
        dwcFile.setFieldsEnclosedBy(reader.quoteChar);
        String[] headers = reader.header;
        int index = 0;
        for (String head : headers) {
            if (head != null && head.length() > 1) {
                try {
                    Term dt = TERM_FACTORY.findTerm(head);
                    ArchiveField field = new ArchiveField(index, dt, null, ArchiveField.DataType.string);
                    if (dwcFile.getId() == null && (dt.equals(DwcTerm.occurrenceID) || dt.equals(DwcTerm.taxonID) || dt.equals(DwcTerm.eventID) || dt.equals(DcTerm.identifier))) {
                        dwcFile.setId(field);
                        if (dwcFile.getRowType() == null && dwcFile.getId().getTerm() != DcTerm.identifier) {
                            if (dt.equals(DwcTerm.occurrenceID)) {
                                dwcFile.setRowType(DwcTerm.Occurrence);
                            } else if (dt.equals(DwcTerm.taxonID)) {
                                dwcFile.setRowType(DwcTerm.Taxon);
                            } else if (dt.equals(DwcTerm.eventID)) {
                                dwcFile.setRowType(DwcTerm.Event);
                            }
                        }
                    }
                    dwcFile.addField(field);
                }
                catch (IllegalArgumentException e) {
                    LOG.warn("Illegal term name >>{}<< found in header, ignore column {}", (Object)head, (Object)index);
                }
            }
            ++index;
        }
        return dwcFile;
    }

    @VisibleForTesting
    protected static void readMetaDescriptor(Archive archive, InputStream metaDescriptor) throws UnsupportedArchiveException {
        try {
            SAXParser p = SAX_FACTORY.newSAXParser();
            MetaHandler mh = new MetaHandler(archive);
            LOG.debug("Reading archive metadata file");
            p.parse((InputStream)new BOMInputStream(metaDescriptor), (DefaultHandler)mh);
        }
        catch (Exception e1) {
            LOG.warn("Exception caught", (Throwable)e1);
            throw new UnsupportedArchiveException(e1);
        }
    }

    private static Archive validateArchive(Archive archive) throws UnsupportedArchiveException {
        ArchiveFactory.validateCoreFile(archive.getCore(), !archive.getExtensions().isEmpty());
        for (ArchiveFile af : archive.getExtensions()) {
            ArchiveFactory.validateExtensionFile(af);
        }
        LOG.debug("Archive contains " + archive.getExtensions().size() + " described extension files");
        LOG.debug("Archive contains " + archive.getCore().getFields().size() + " core properties");
        return archive;
    }

    private static void validateCoreFile(ArchiveFile f, boolean hasExtensions) throws UnsupportedArchiveException {
        if (hasExtensions && f.getId() == null) {
            LOG.warn("DwC-A core data file " + f.getTitle() + " is lacking an id column. No extensions allowed in this case");
        }
        ArchiveFactory.validateFile(f);
    }

    private static void validateExtensionFile(ArchiveFile f) throws UnsupportedArchiveException {
        if (f.getId() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires an id or foreign key to the core id");
        }
        ArchiveFactory.validateFile(f);
    }

    private static void validateFile(ArchiveFile f) throws UnsupportedArchiveException {
        if (f == null) {
            throw new UnsupportedArchiveException("DwC-A data file is NULL");
        }
        if (f.getLocationFile() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires a location");
        }
        if (f.getEncoding() == null) {
            throw new UnsupportedArchiveException("DwC-A data file " + f.getTitle() + " requires a character encoding");
        }
    }

    static {
        SAX_FACTORY.setNamespaceAware(true);
        SAX_FACTORY.setValidating(false);
    }

    static class MetaHandler
    extends SimpleSaxHandler {
        private static final String NS_DWCA = "http://rs.tdwg.org/dwc/text/";
        private Archive archive;
        private ArchiveFile af;

        protected MetaHandler(Archive archive) {
            this.archive = archive;
        }

        private static Character getFirstChar(String x) throws UnsupportedArchiveException {
            if (x == null || x.length() == 0) {
                return null;
            }
            if (x.length() == 1) {
                return Character.valueOf(x.charAt(0));
            }
            if (x.equalsIgnoreCase("\\t")) {
                return Character.valueOf('\t');
            }
            if (x.equalsIgnoreCase("\\n")) {
                return Character.valueOf('\n');
            }
            if (x.equalsIgnoreCase("\\r")) {
                return Character.valueOf('\r');
            }
            if (x.length() > 1) {
                throw new UnsupportedArchiveException("Only darwin core archives with a single quotation character are supported, but found >>>" + x + "<<<");
            }
            return Character.valueOf(' ');
        }

        private static String unescapeBackslash(String x) {
            if (x == null || x.length() == 0) {
                return null;
            }
            return x.replaceAll("\\\\t", String.valueOf('\t')).replaceAll("\\\\n", String.valueOf('\n')).replaceAll("\\\\r", String.valueOf('\r')).replaceAll("\\\\f", String.valueOf('\f'));
        }

        private ArchiveFile buildArchiveFile(Attributes attr) throws UnsupportedArchiveException {
            ArchiveFile dwcFile = new ArchiveFile();
            if (this.getAttr(attr, "encoding") != null) {
                dwcFile.setEncoding(this.getAttr(attr, "encoding"));
            }
            if (this.getAttr(attr, "fieldsTerminatedBy") != null) {
                dwcFile.setFieldsTerminatedBy(MetaHandler.unescapeBackslash(this.getAttr(attr, "fieldsTerminatedBy")));
            }
            if (this.getAttr(attr, "fieldsEnclosedBy") != null) {
                dwcFile.setFieldsEnclosedBy(MetaHandler.getFirstChar(this.getAttr(attr, "fieldsEnclosedBy")));
            }
            if (this.getAttr(attr, "linesTerminatedBy") != null) {
                dwcFile.setLinesTerminatedBy(MetaHandler.unescapeBackslash(this.getAttr(attr, "linesTerminatedBy")));
            }
            if (this.getAttr(attr, "rowType") != null) {
                dwcFile.setRowType(TERM_FACTORY.findTerm(this.getAttr(attr, "rowType")));
            }
            String ignoreHeaderLines = this.getAttr(attr, "ignoreHeaderLines");
            try {
                dwcFile.setIgnoreHeaderLines(Integer.parseInt(ignoreHeaderLines));
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
            return dwcFile;
        }

        private ArchiveField buildField(Attributes attributes) {
            Term term = TERM_FACTORY.findTerm(this.getAttr(attributes, "term"));
            String defaultValue = this.getAttr(attributes, "default");
            String vocabulary = this.getAttr(attributes, "vocabulary");
            ArchiveField.DataType type = ArchiveField.DataType.findByXmlSchemaType(this.getAttr(attributes, "type"));
            if (type == null) {
                type = ArchiveField.DataType.string;
            }
            String indexAsString = this.getAttr(attributes, "index");
            Integer index = null;
            if (indexAsString != null) {
                try {
                    index = Integer.parseInt(indexAsString);
                }
                catch (NumberFormatException e) {
                    throw new UnsupportedArchiveException(e);
                }
            }
            String delimiter = this.getAttr(attributes, "delimitedBy");
            return new ArchiveField(index, term, defaultValue, type, delimiter, vocabulary);
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            super.endElement(uri, localName, qName);
            if (!localName.equalsIgnoreCase("archive")) {
                if (localName.equalsIgnoreCase("core")) {
                    this.archive.setCore(this.af);
                } else if (localName.equalsIgnoreCase("extension")) {
                    if (this.af.getId() != null && this.af.getId().getIndex() != null) {
                        this.archive.addExtension(this.af);
                    } else {
                        this.log.warn("Skipping extension [" + this.af.getRowType() + "] with no index attribute");
                    }
                } else if (localName.equalsIgnoreCase("location")) {
                    this.af.addLocation(this.content);
                }
            }
        }

        private String getAttr(Attributes attributes, String key) {
            String val = null;
            if (attributes != null && (val = attributes.getValue("", key)) == null) {
                val = attributes.getValue(NS_DWCA, key);
            }
            return Strings.isNullOrEmpty(val) ? null : val;
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            super.startElement(uri, localName, qName, attributes);
            if (localName.equalsIgnoreCase("archive") || localName.equalsIgnoreCase("stararchive")) {
                this.archive.setMetadataLocation(this.getAttr(attributes, "metadata"));
            } else if (localName.equalsIgnoreCase("core") || localName.equalsIgnoreCase("extension")) {
                this.af = new ArchiveFile();
                if (localName.equalsIgnoreCase("core") || localName.equalsIgnoreCase("extension")) {
                    this.af = this.buildArchiveFile(attributes);
                }
            } else if (localName.equalsIgnoreCase("coreid") || localName.equalsIgnoreCase("id")) {
                ArchiveField field = this.buildField(attributes);
                if (this.af != null) {
                    this.af.setId(field);
                } else {
                    this.log.warn(localName + " field found outside of an archive file");
                }
            } else if (localName.equalsIgnoreCase("field")) {
                ArchiveField field = this.buildField(attributes);
                if (this.af != null) {
                    this.af.addField(field);
                } else {
                    this.log.warn("field found outside of an archive file");
                }
            }
        }
    }
}

