/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.ws.server.filter;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import com.sun.jersey.core.util.Base64;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import java.security.Principal;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import org.gbif.api.model.common.ExtendedPrincipal;
import org.gbif.api.model.common.GbifUser;
import org.gbif.api.model.common.GbifUserPrincipal;
import org.gbif.api.service.common.IdentityAccessService;
import org.gbif.ws.security.GbifAuthService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IdentityFilter
implements ContainerRequestFilter {
    private static final Logger LOG = LoggerFactory.getLogger(IdentityFilter.class);
    private static final Pattern COLON_PATTERN = Pattern.compile(":");
    private final IdentityAccessService identityAccessService;
    private final GbifAuthService authService;
    private static final String GBIF_SCHEME_PREFIX = "GBIF ";
    private static final String BASIC_SCHEME_PREFIX = "Basic ";

    @Inject
    public IdentityFilter(@NotNull IdentityAccessService identityAccessService, @Nullable GbifAuthService authService) {
        Objects.requireNonNull(identityAccessService, "identityAccessService shall be provided");
        this.identityAccessService = identityAccessService;
        this.authService = authService;
    }

    public ContainerRequest filter(ContainerRequest request) {
        Authorizer authorizer = this.authenticate(request);
        if (authorizer == null) {
            authorizer = Authorizer.getAnonymous(request.isSecure());
        }
        request.setSecurityContext((SecurityContext)authorizer);
        return request;
    }

    private Authorizer authenticate(ContainerRequest request) {
        String authentication = request.getHeaderValue("Authorization");
        if (authentication != null) {
            if (authentication.startsWith(BASIC_SCHEME_PREFIX)) {
                return this.basicAuthentication(authentication.substring(BASIC_SCHEME_PREFIX.length()), request.isSecure());
            }
            if (authentication.startsWith(GBIF_SCHEME_PREFIX)) {
                return this.gbifAuthentication(request);
            }
        }
        return Authorizer.getAnonymous(request.isSecure());
    }

    private Authorizer basicAuthentication(String authentication, boolean isSecure) {
        String[] values = COLON_PATTERN.split(Base64.base64Decode((String)authentication));
        if (values.length < 2) {
            LOG.warn("Invalid syntax for username and password: {}", (Object)authentication);
            throw new WebApplicationException(Response.Status.BAD_REQUEST);
        }
        String username = values[0];
        String password = values[1];
        if (username == null || password == null) {
            LOG.warn("Missing basic authentication username or password: {}", (Object)authentication);
            throw new WebApplicationException(Response.Status.BAD_REQUEST);
        }
        try {
            UUID.fromString(username);
            return null;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            GbifUser user = this.identityAccessService.authenticate(username, password);
            if (user == null) {
                throw new WebApplicationException(Response.Status.UNAUTHORIZED);
            }
            LOG.debug("Authenticating user {} via scheme {}", (Object)username, (Object)"BASIC");
            return Authorizer.getAuthorizer(user, "BASIC", isSecure);
        }
    }

    private Authorizer gbifAuthentication(ContainerRequest request) {
        String username = request.getHeaderValue("x-gbif-user");
        if (Strings.isNullOrEmpty((String)username)) {
            LOG.warn("Missing gbif username header {}", (Object)"x-gbif-user");
            throw new WebApplicationException(Response.Status.BAD_REQUEST);
        }
        if (this.authService == null) {
            LOG.warn("No GbifAuthService defined.");
            throw new WebApplicationException(Response.Status.UNAUTHORIZED);
        }
        if (!this.authService.isValidRequest(request)) {
            LOG.warn("Invalid GBIF authenticated request");
            throw new WebApplicationException(Response.Status.UNAUTHORIZED);
        }
        LOG.debug("Authenticating user {} via scheme {}", (Object)username, (Object)"GBIF");
        if (this.identityAccessService == null) {
            LOG.debug("No identityService configured! No roles assigned, using anonymous user instead.");
            return Authorizer.getAnonymous(request.isSecure());
        }
        GbifUser user = this.identityAccessService.get(username);
        return user == null ? Authorizer.getAnonymous(request.isSecure()) : Authorizer.getAuthorizer(user, "GBIF", Optional.ofNullable(request.getSecurityContext()).map(sc -> request.isSecure()).orElse(Boolean.FALSE));
    }

    private static class Authorizer
    implements SecurityContext {
        private final ExtendedPrincipal principal;
        private final String authenticationScheme;
        private final boolean isSecure;

        private Authorizer(ExtendedPrincipal principal, String authenticationScheme, boolean isSecure) {
            this.principal = principal;
            this.authenticationScheme = authenticationScheme;
            this.isSecure = isSecure;
        }

        static Authorizer getAuthorizer(GbifUser user, String authenticationScheme, boolean isSecure) {
            return new Authorizer((ExtendedPrincipal)new GbifUserPrincipal(user), authenticationScheme, isSecure);
        }

        static Authorizer getAnonymous(boolean isSecure) {
            return new Authorizer(null, "", isSecure);
        }

        public String getAuthenticationScheme() {
            return this.authenticationScheme;
        }

        public Principal getUserPrincipal() {
            return this.principal;
        }

        public boolean isSecure() {
            return this.isSecure;
        }

        public boolean isUserInRole(String role) {
            return this.principal != null && this.principal.hasRole(role);
        }
    }
}

