/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.ws.security;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.inject.Singleton;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.core.util.Base64;
import com.sun.jersey.spi.container.ContainerRequest;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.regex.Pattern;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.ws.rs.core.MultivaluedMap;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.map.ObjectMapper;
import org.gbif.ws.json.JacksonJsonContextResolver;
import org.gbif.ws.security.RequestHeaderAccessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class GbifAuthService {
    private static final Logger LOG = LoggerFactory.getLogger(GbifAuthService.class);
    private static final String ALGORITHM = "HmacSHA1";
    private static final String CHAR_ENCODING = "UTF8";
    public static final String HEADER_AUTHORIZATION = "Authorization";
    public static final String HEADER_CONTENT_TYPE = "Content-Type";
    public static final String HEADER_CONTENT_MD5 = "Content-MD5";
    public static final String GBIF_SCHEME = "GBIF";
    public static final String HEADER_GBIF_USER = "x-gbif-user";
    public static final String HEADER_ORIGINAL_REQUEST_URL = "x-url";
    private static final char NEWLINE = '\n';
    private static final Pattern COLON_PATTERN = Pattern.compile(":");
    private final ImmutableMap<String, String> keyStore;
    private final String appKey;
    private final ObjectMapper mapper = new JacksonJsonContextResolver().getContext(null);

    private GbifAuthService(Map<String, String> appKeys, String appKey) {
        this.keyStore = ImmutableMap.copyOf(appKeys);
        this.appKey = appKey;
        LOG.info("Initialised appkey store with {} keys", (Object)this.keyStore.size());
    }

    public static GbifAuthService multiKeyAuthService(Map<String, String> appKeys) {
        return new GbifAuthService(appKeys, null);
    }

    public static GbifAuthService singleKeyAuthService(String appKey, String secret) {
        LOG.info("Initialising auth service with key {}", (Object)appKey);
        return new GbifAuthService((Map<String, String>)ImmutableMap.of((Object)appKey, (Object)secret), appKey);
    }

    private static String buildStringToSign(ContainerRequest request) {
        StringBuilder sb = new StringBuilder();
        sb.append(request.getMethod());
        sb.append('\n');
        if (request.getRequestHeaders().containsKey((Object)HEADER_ORIGINAL_REQUEST_URL)) {
            sb.append((String)request.getRequestHeaders().getFirst((Object)HEADER_ORIGINAL_REQUEST_URL));
        } else {
            sb.append(GbifAuthService.getCanonicalizedPath(request.getRequestUri()));
        }
        GbifAuthService.appendHeader(sb, request.getRequestHeaders(), HEADER_CONTENT_TYPE, false);
        GbifAuthService.appendHeader(sb, request.getRequestHeaders(), HEADER_CONTENT_MD5, true);
        GbifAuthService.appendHeader(sb, request.getRequestHeaders(), HEADER_GBIF_USER, true);
        return sb.toString();
    }

    private static String buildStringToSign(ClientRequest request) {
        StringBuilder sb = new StringBuilder();
        sb.append(request.getMethod());
        sb.append('\n');
        sb.append(GbifAuthService.getCanonicalizedPath(request.getURI()));
        GbifAuthService.appendHeader(sb, request.getHeaders(), HEADER_CONTENT_TYPE, false);
        GbifAuthService.appendHeader(sb, request.getHeaders(), HEADER_CONTENT_MD5, true);
        GbifAuthService.appendHeader(sb, request.getHeaders(), HEADER_GBIF_USER, true);
        return sb.toString();
    }

    private static void appendHeader(StringBuilder sb, MultivaluedMap<String, ?> request, String header, boolean caseSensitive) {
        if (request.containsKey((Object)header)) {
            sb.append('\n');
            if (caseSensitive) {
                sb.append(request.getFirst((Object)header));
            } else {
                sb.append(request.getFirst((Object)header).toString().toLowerCase());
            }
        }
    }

    private static String getCanonicalizedPath(URI uri) {
        return uri.normalize().getPath();
    }

    private static String buildAuthHeader(String applicationKey, String signature) {
        return "GBIF " + applicationKey + ':' + signature;
    }

    private String buildSignature(String stringToSign, String secretKey) {
        try {
            Mac mac = Mac.getInstance(ALGORITHM);
            SecretKeySpec secret = new SecretKeySpec(secretKey.getBytes(Charset.forName(CHAR_ENCODING)), ALGORITHM);
            mac.init(secret);
            byte[] digest = mac.doFinal(stringToSign.getBytes());
            return new String(Base64.encode((byte[])digest), "ASCII");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Cant find HmacSHA1 message digester", e);
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unsupported character encoding UTF8", e);
        }
        catch (InvalidKeyException e) {
            throw new RuntimeException("Invalid secret key " + secretKey, e);
        }
    }

    public void signRequest(String username, ClientRequest request) {
        Preconditions.checkNotNull((Object)this.appKey, (Object)"To sign request a single application key is required");
        request.getHeaders().putSingle((Object)HEADER_GBIF_USER, (Object)username);
        request.getHeaders().putSingle((Object)HEADER_ORIGINAL_REQUEST_URL, (Object)GbifAuthService.getCanonicalizedPath(request.getURI()));
        if (request.getEntity() != null) {
            request.getHeaders().putSingle((Object)HEADER_CONTENT_MD5, (Object)this.buildContentMD5(request.getEntity()));
        }
        String stringToSign = GbifAuthService.buildStringToSign(request);
        String secretKey = this.getPrivateKey(this.appKey);
        if (secretKey == null) {
            LOG.warn("Skip signing request with unknown application key: {}", (Object)this.appKey);
            return;
        }
        String signature = this.buildSignature(stringToSign, secretKey);
        String header = GbifAuthService.buildAuthHeader(this.appKey, signature);
        LOG.debug("Adding authentication header to request {} for proxied user {} : {}", new Object[]{request.getURI(), username, header});
        request.getHeaders().putSingle((Object)HEADER_AUTHORIZATION, (Object)header);
    }

    private String buildContentMD5(Object entity) {
        try {
            byte[] content = this.mapper.writeValueAsBytes(entity);
            return new String(Base64.encode((byte[])DigestUtils.md5((byte[])content)), "ASCII");
        }
        catch (IOException e) {
            LOG.error("Failed to serialize http entity [{}]", entity);
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static String getAppKeyFromRequest(RequestHeaderAccessor requestHeaderAccessor) {
        String[] values;
        if (StringUtils.startsWith((CharSequence)requestHeaderAccessor.getHeader(HEADER_AUTHORIZATION), (CharSequence)"GBIF ") && (values = COLON_PATTERN.split(requestHeaderAccessor.getHeader(HEADER_AUTHORIZATION).substring(5), 2)).length == 2) {
            return values[0];
        }
        return null;
    }

    public boolean isValidRequest(ContainerRequest request) {
        String authHeader = request.getHeaderValue(HEADER_AUTHORIZATION);
        if (Strings.isNullOrEmpty((String)authHeader) || !authHeader.startsWith("GBIF ")) {
            LOG.info("Authorization header is no GBIF scheme");
            return false;
        }
        String[] values = COLON_PATTERN.split(authHeader.substring(5), 2);
        if (values.length < 2) {
            LOG.warn("Invalid syntax for application key and signature: {}", (Object)authHeader);
            return false;
        }
        String appKey = values[0];
        String signatureFound = values[1];
        if (appKey == null || signatureFound == null) {
            LOG.warn("Authentication header missing applicationKey or signature: {}", (Object)authHeader);
            return false;
        }
        String secretKey = this.getPrivateKey(appKey);
        if (secretKey == null) {
            LOG.warn("Unknown application key: {}", (Object)appKey);
            return false;
        }
        String stringToSign = GbifAuthService.buildStringToSign(request);
        String signature = this.buildSignature(stringToSign, secretKey);
        if (signatureFound.equals(signature)) {
            LOG.debug("Trusted application with matching signatures");
            return true;
        }
        LOG.info("Invalid signature: {}", (Object)authHeader);
        LOG.debug("StringToSign: {}", (Object)stringToSign);
        return false;
    }

    private String getPrivateKey(String applicationKey) {
        return (String)this.keyStore.get((Object)applicationKey);
    }
}

