/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.api.util;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.Polygon;
import com.vividsolutions.jts.io.ParseException;
import com.vividsolutions.jts.io.WKTReader;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.gbif.api.model.common.search.SearchParameter;
import org.gbif.api.model.occurrence.search.OccurrenceSearchParameter;
import org.gbif.api.util.IsoDateParsingUtils;
import org.gbif.api.util.VocabularyUtils;
import org.gbif.api.vocabulary.Country;
import org.gbif.api.vocabulary.Language;

public class SearchTypeValidator {
    private static final Pattern BOOLEAN = Pattern.compile("^(true|false)$", 2);
    private static final String DEC = "-?\\d+(?:\\.\\d+)?";
    private static final String DECIMAL = "(-?\\d+(?:\\.\\d+)?)";
    private static final String DECIMAL_OR_WILDCARD = "(-?\\d+(?:\\.\\d+)?|\\*)";
    private static final Range<Double> LATITUDE_RNG = Range.closed((Comparable)Double.valueOf(-90.0), (Comparable)Double.valueOf(90.0));
    private static final Range<Double> LONGITUDE_RNG = Range.closed((Comparable)Double.valueOf(-180.0), (Comparable)Double.valueOf(180.0));
    private static final String LATITUDE_ERROR_MSG = "%s is not valid value, latitude must be between -90 and 90.";
    private static final String LONGITUDE_ERROR_MSG = "%s is not valid value, longitude must be between -180 and 180.";
    private static final String WILD_CARD = "*";
    private static final Pattern DECIMAL_RANGE_PATTERN = Pattern.compile("^(-?\\d+(?:\\.\\d+)?|\\*)\\s*,\\s*(-?\\d+(?:\\.\\d+)?|\\*)$", 2);
    private static final String WKT_COORD = "\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*";
    private static final String WKT_LINE = "\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*";
    private static final String WKT_POLYGON = "\\(\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*(?:,\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*)*\\)";
    private static final List<Pattern> WKT_PATTERNS = ImmutableList.of((Object)Pattern.compile("^POINT\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*\\)$", 2), (Object)Pattern.compile("^(?:LINESTRING|LINEARRING)\\s*\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*$", 2), (Object)Pattern.compile("^POLYGON\\s*\\(\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*(?:,\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*)*\\)$", 2), (Object)Pattern.compile("^MULTIPOLYGON\\s*\\(\\(\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*(?:,\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*)*\\)(?:,\\(\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*(?:,\\s*\\(\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*(?:,\\s*(-?\\d+(?:\\.\\d+)?)\\s+(-?\\d+(?:\\.\\d+)?)\\s*)*\\)\\s*)*\\))*\\)$", 2));

    private SearchTypeValidator() {
    }

    public static <T extends Comparable<?>> Range<T> buildRange(T lower, T upper) {
        if (lower == null && upper != null) {
            return Range.atMost(upper);
        }
        if (lower != null && upper == null) {
            return Range.atLeast(lower);
        }
        if (lower == null && upper == null) {
            return Range.all();
        }
        return Range.closed(lower, upper);
    }

    public static boolean isRange(String value) {
        if (!Strings.isNullOrEmpty((String)value)) {
            if (DECIMAL_RANGE_PATTERN.matcher(value).find()) {
                return true;
            }
            try {
                IsoDateParsingUtils.parseDateRange(value);
                return true;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return false;
    }

    public static Range<Date> parseDateRange(String value) {
        return IsoDateParsingUtils.parseDateRange(value);
    }

    public static Range<Double> parseDecimalRange(String value) {
        Matcher m;
        if (!Strings.isNullOrEmpty((String)value) && (m = DECIMAL_RANGE_PATTERN.matcher(value)).find()) {
            return SearchTypeValidator.buildRange(SearchTypeValidator.parseDouble(m.group(1)), SearchTypeValidator.parseDouble(m.group(2)));
        }
        throw new IllegalArgumentException("Invalid decimal range: " + value);
    }

    public static Range<Integer> parseIntegerRange(String value) {
        Matcher m;
        if (!Strings.isNullOrEmpty((String)value) && (m = DECIMAL_RANGE_PATTERN.matcher(value)).find()) {
            return SearchTypeValidator.buildRange(SearchTypeValidator.parseInteger(m.group(1)), SearchTypeValidator.parseInteger(m.group(2)));
        }
        throw new IllegalArgumentException("Invalid integer range: " + value);
    }

    public static void validate(SearchParameter param, String value) throws IllegalArgumentException {
        Class<?> pType = param.type();
        try {
            if (OccurrenceSearchParameter.GEOMETRY == param) {
                SearchTypeValidator.validateGeometry(value);
            }
            if (!WILD_CARD.equalsIgnoreCase(Strings.nullToEmpty((String)value).trim())) {
                if (OccurrenceSearchParameter.DECIMAL_LATITUDE == param) {
                    SearchTypeValidator.validateLatitude(value);
                } else if (OccurrenceSearchParameter.DECIMAL_LONGITUDE == param) {
                    SearchTypeValidator.validateLongitude(value);
                } else if (UUID.class.isAssignableFrom(pType)) {
                    UUID.fromString(value);
                } else if (Double.class.isAssignableFrom(pType)) {
                    SearchTypeValidator.validateDouble(value);
                } else if (Integer.class.isAssignableFrom(pType)) {
                    Collection<Integer> intsFound = SearchTypeValidator.validateInteger(value);
                    if (OccurrenceSearchParameter.MONTH == param) {
                        SearchTypeValidator.validateMonth(intsFound);
                    }
                } else if (Boolean.class.isAssignableFrom(pType)) {
                    if (!BOOLEAN.matcher(value).find()) {
                        throw new IllegalArgumentException("Value " + value + " is no valid boolean");
                    }
                } else if (Country.class.isAssignableFrom(pType)) {
                    Preconditions.checkNotNull((Object)((Object)Country.fromIsoCode(value)));
                } else if (Language.class.isAssignableFrom(pType)) {
                    Preconditions.checkNotNull((Object)((Object)Language.fromIsoCode(value)));
                } else if (Enum.class.isAssignableFrom(pType)) {
                    Class<?> eType = pType;
                    Preconditions.checkNotNull(VocabularyUtils.lookupEnum(value, eType));
                } else if (Date.class.isAssignableFrom(pType) || Temporal.class.isAssignableFrom(pType)) {
                    SearchTypeValidator.validateDate(value);
                } else if (!String.class.isAssignableFrom(pType)) {
                    throw new IllegalArgumentException("Unknown SearchParameter data type " + pType.getCanonicalName());
                }
            }
        }
        catch (NullPointerException e) {
            throw new IllegalArgumentException("Value " + value + " invalid for filter parameter " + param, e);
        }
    }

    private static Double parseDouble(String d) {
        if (WILD_CARD.equals(d)) {
            return null;
        }
        return Double.parseDouble(d);
    }

    private static Integer parseInteger(String d) {
        if (WILD_CARD.equals(d)) {
            return null;
        }
        return Integer.parseInt(d);
    }

    private static void validateDate(String value) {
        if (SearchTypeValidator.isRange(value)) {
            IsoDateParsingUtils.parseDateRange(value);
        } else {
            IsoDateParsingUtils.parseDate(value);
        }
    }

    private static void validateDouble(String value) {
        if (Strings.isNullOrEmpty((String)value)) {
            throw new IllegalArgumentException("Double cannot be null or empty");
        }
        try {
            Double.parseDouble(value);
        }
        catch (NumberFormatException e) {
            SearchTypeValidator.parseDecimalRange(value);
        }
    }

    private static void validateDoubleInRange(String value, Range<Double> range, String errorMsg) {
        if (Strings.isNullOrEmpty((String)value)) {
            throw new IllegalArgumentException("Double cannot be null or empty");
        }
        try {
            Double doubleValue = Double.parseDouble(value);
            if (!range.contains((Comparable)doubleValue)) {
                throw new IllegalArgumentException(String.format(errorMsg, value));
            }
        }
        catch (NumberFormatException e) {
            if (SearchTypeValidator.isRange(value)) {
                Range<Double> rangeValue = SearchTypeValidator.parseDecimalRange(value);
                if (!range.encloses(rangeValue)) {
                    throw new IllegalArgumentException(String.format(errorMsg, value));
                }
            }
            throw new IllegalArgumentException("Argument is not a valid number");
        }
    }

    private static void validateGeometry(String wellKnownText) {
        SearchTypeValidator.validateGeometrySyntax(wellKnownText);
        try {
            Geometry geometry = new WKTReader().read(wellKnownText);
            if (geometry instanceof Polygon && (!geometry.isValid() || geometry.getArea() == 0.0)) {
                throw new IllegalArgumentException("Invalid polygon " + wellKnownText);
            }
        }
        catch (ParseException e) {
            throw new IllegalArgumentException("Invalid simple WKT: " + wellKnownText);
        }
    }

    private static void validateGeometrySyntax(String wellKnownText) {
        if (Strings.isNullOrEmpty((String)wellKnownText)) {
            throw new IllegalArgumentException("Well Known Text cannot be empty or null");
        }
        for (Pattern regex : WKT_PATTERNS) {
            if (!regex.matcher(wellKnownText).find()) continue;
            return;
        }
        throw new IllegalArgumentException("Invalid simple WKT: " + wellKnownText);
    }

    private static Collection<Integer> validateInteger(String value) {
        if (Strings.isNullOrEmpty((String)value)) {
            throw new IllegalArgumentException("Integer cannot be null or empty");
        }
        try {
            return Lists.newArrayList((Object[])new Integer[]{Integer.parseInt(value)});
        }
        catch (NumberFormatException e) {
            Range<Integer> range = SearchTypeValidator.parseIntegerRange(value);
            ArrayList ints = Lists.newArrayList();
            if (range.hasLowerBound()) {
                ints.add(range.lowerEndpoint());
            }
            if (range.hasUpperBound()) {
                ints.add(range.upperEndpoint());
            }
            return ints;
        }
    }

    private static void validateLatitude(String value) {
        SearchTypeValidator.validateDoubleInRange(value, LATITUDE_RNG, LATITUDE_ERROR_MSG);
    }

    private static void validateLongitude(String value) {
        SearchTypeValidator.validateDoubleInRange(value, LONGITUDE_RNG, LONGITUDE_ERROR_MSG);
    }

    private static void validateMonth(Collection<Integer> months) {
        for (Integer month : months) {
            if (month == null || month >= 1 && month <= 12) continue;
            throw new IllegalArgumentException("Month needs to be between 1 - 12");
        }
    }
}

