/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.registry.doi;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.bind.JAXBException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.gbif.api.model.common.DOI;
import org.gbif.api.model.common.GbifUser;
import org.gbif.api.model.occurrence.Download;
import org.gbif.api.model.registry.Contact;
import org.gbif.api.model.registry.Dataset;
import org.gbif.api.model.registry.DatasetOccurrenceDownloadUsage;
import org.gbif.api.model.registry.Organization;
import org.gbif.api.model.registry.eml.KeywordCollection;
import org.gbif.api.model.registry.eml.geospatial.GeospatialCoverage;
import org.gbif.api.vocabulary.ContactType;
import org.gbif.api.vocabulary.IdentifierType;
import org.gbif.api.vocabulary.Language;
import org.gbif.api.vocabulary.License;
import org.gbif.doi.metadata.datacite.ContributorType;
import org.gbif.doi.metadata.datacite.DataCiteMetadata;
import org.gbif.doi.metadata.datacite.DateType;
import org.gbif.doi.metadata.datacite.DescriptionType;
import org.gbif.doi.metadata.datacite.ObjectFactory;
import org.gbif.doi.metadata.datacite.RelatedIdentifierType;
import org.gbif.doi.metadata.datacite.RelationType;
import org.gbif.doi.metadata.datacite.ResourceType;
import org.gbif.doi.service.InvalidMetadataException;
import org.gbif.doi.service.datacite.DataCiteValidator;
import org.gbif.occurrence.query.HumanFilterBuilder;
import org.gbif.occurrence.query.TitleLookup;
import org.gbif.registry.metadata.contact.ContactAdapter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataCiteConverter {
    private static final Logger LOG = LoggerFactory.getLogger(DataCiteConverter.class);
    private static final ObjectFactory FACTORY = new ObjectFactory();
    public static final String ORCID_NAME_IDENTIFIER_SCHEME = "ORCID";
    public static final String RESEARCHERID_NAME_IDENTIFIER_SCHEME = "ResearcherID";
    private static Map<ContactType, ContributorType> REGISTRY_DATACITE_ROLE_MAPPING = ImmutableMap.builder().put(ContactType.EDITOR, ContributorType.EDITOR).put(ContactType.PUBLISHER, ContributorType.EDITOR).put(ContactType.CONTENT_PROVIDER, ContributorType.DATA_COLLECTOR).put(ContactType.CUSTODIAN_STEWARD, ContributorType.DATA_MANAGER).put(ContactType.CURATOR, ContributorType.DATA_CURATOR).put(ContactType.METADATA_AUTHOR, ContributorType.DATA_CURATOR).put(ContactType.DISTRIBUTOR, ContributorType.DISTRIBUTOR).put(ContactType.OWNER, ContributorType.RIGHTS_HOLDER).put(ContactType.POINT_OF_CONTACT, ContributorType.CONTACT_PERSON).put(ContactType.PRINCIPAL_INVESTIGATOR, ContributorType.PROJECT_LEADER).put(ContactType.ORIGINATOR, ContributorType.DATA_COLLECTOR).put(ContactType.PROCESSOR, ContributorType.PRODUCER).put(ContactType.PROGRAMMER, ContributorType.PRODUCER).build();
    public static final Map<Pattern, String> SUPPORTED_SCHEMES = ImmutableMap.of(Pattern.compile("^(http[s]?:\\/\\/orcid.org\\/)([\\d\\-]+$)"), "ORCID");
    private static final String DOWNLOAD_TITLE = "GBIF Occurrence Download";
    protected static final String GBIF_PUBLISHER = "The Global Biodiversity Information Facility";
    protected static final License DEFAULT_DOWNLOAD_LICENSE = License.CC_BY_NC_4_0;
    private static final String LICENSE_INFO = "Data from some individual datasets included in this download may be licensed under less restrictive terms.";
    protected static final String ENGLISH = Language.ENGLISH.getIso3LetterCode();
    private static final String DWCA_FORMAT = "Darwin Core Archive";

    private static String fdate(Date date) {
        return DateFormatUtils.ISO_DATE_FORMAT.format(date);
    }

    public static DataCiteMetadata convert(Dataset d, Organization publisher) {
        DataCiteMetadata.Builder<Void> b = DataCiteMetadata.builder().withTitles().withTitle(DataCiteMetadata.Titles.Title.builder().withValue(d.getTitle()).build()).end().withPublisher(publisher.getTitle()).withPublicationYear(DataCiteConverter.getYear(new Date())).withResourceType().withResourceTypeGeneral(ResourceType.DATASET).withValue(d.getType().name()).end().withRelatedIdentifiers().end();
        if (d.getCreated() != null) {
            b.withPublicationYear(DataCiteConverter.getYear(d.getModified())).withDates().addDate().withDateType(DateType.CREATED).withValue(DataCiteConverter.fdate(d.getCreated())).end().addDate().withDateType(DateType.UPDATED).withValue(DataCiteConverter.fdate(d.getModified())).end().end();
        }
        boolean creatorFound = false;
        if (d.getContacts() != null && !d.getContacts().isEmpty()) {
            ContactAdapter contactAdapter = new ContactAdapter(d.getContacts());
            List resourceCreators = contactAdapter.getCreators();
            DataCiteMetadata.Creators.Builder<DataCiteMetadata.Builder<Void>> creatorsBuilder = b.withCreators();
            for (Contact resourceCreator : resourceCreators) {
                DataCiteMetadata.Creators.Creator creator = DataCiteConverter.toDataCiteCreator(resourceCreator);
                if (creator == null) continue;
                creatorsBuilder.addCreator(creator).end();
                creatorFound = true;
            }
            creatorsBuilder.end();
            ArrayList<Contact> contributors = Lists.newArrayList(d.getContacts());
            contributors.removeAll(resourceCreators);
            if (!contributors.isEmpty()) {
                DataCiteMetadata.Contributors.Builder<DataCiteMetadata.Builder<Void>> contributorsBuilder = b.withContributors();
                for (Contact contact : contributors) {
                    DataCiteMetadata.Contributors.Contributor contributor = DataCiteConverter.toDataCiteContributor(contact);
                    if (contributor == null) continue;
                    contributorsBuilder.addContributor(contributor).end();
                }
                contributorsBuilder.end();
            }
        }
        if (!creatorFound) {
            b.withCreators().addCreator(DataCiteConverter.getDefaultGBIFDataCiteCreator(d.getCreatedBy())).end().end();
        }
        if (d.getPubDate() != null) {
            b.withPublicationYear(DataCiteConverter.getYear(d.getPubDate()));
        }
        if (d.getModified() != null) {
            b.withDates().addDate().withDateType(DateType.UPDATED).withValue(DataCiteConverter.fdate(d.getModified()));
        }
        if (d.getDoi() != null) {
            b.withIdentifier().withIdentifierType(IdentifierType.DOI.name()).withValue(d.getDoi().getDoiName());
            if (d.getKey() != null) {
                b.withAlternateIdentifiers().addAlternateIdentifier().withAlternateIdentifierType("UUID").withValue(d.getKey().toString());
            }
        } else if (d.getKey() != null) {
            b.withIdentifier().withIdentifierType("UUID").withValue(d.getKey().toString());
        }
        if (!Strings.isNullOrEmpty(d.getDescription())) {
            b.withDescriptions().addDescription().addContent(new Serializable[]{d.getDescription()}).withDescriptionType(DescriptionType.ABSTRACT);
        }
        if (d.getDataLanguage() != null) {
            b.withLanguage(d.getDataLanguage().getIso3LetterCode());
        }
        if (d.getLicense() != null && d.getLicense().isConcrete()) {
            b.withRightsList().addRights().withRightsURI(d.getLicense().getLicenseUrl()).withValue(d.getLicense().getLicenseTitle());
        } else if (!Strings.isNullOrEmpty(d.getRights())) {
            b.withRightsList().addRights().withValue(d.getRights()).end();
        }
        HashSet<DataCiteMetadata.Subjects.Subject> subjects = Sets.newHashSet();
        for (KeywordCollection kcol : d.getKeywordCollections()) {
            for (String k : kcol.getKeywords()) {
                if (Strings.isNullOrEmpty(k)) continue;
                DataCiteMetadata.Subjects.Subject s = DataCiteMetadata.Subjects.Subject.builder().withValue(k).build();
                if (!Strings.isNullOrEmpty(kcol.getThesaurus())) {
                    s.setSubjectScheme(kcol.getThesaurus());
                }
                subjects.add(s);
            }
        }
        for (GeospatialCoverage gc : d.getGeographicCoverages()) {
            if (gc.getBoundingBox() == null) continue;
            b.withGeoLocations().addGeoLocation().addGeoLocationBox(gc.getBoundingBox().getMinLatitude(), gc.getBoundingBox().getMinLongitude(), gc.getBoundingBox().getMaxLatitude(), gc.getBoundingBox().getMaxLongitude());
        }
        return b.build();
    }

    private static DataCiteMetadata truncateDescriptionDCM(DOI doi, String xml, URI target) throws InvalidMetadataException {
        try {
            DataCiteMetadata dm = DataCiteValidator.fromXml(xml);
            String description = Joiner.on("\n").join(dm.getDescriptions().getDescription().get(0).getContent());
            dm.setDescriptions(DataCiteMetadata.Descriptions.builder().addDescription().withDescriptionType(DescriptionType.ABSTRACT).withLang(ENGLISH).addContent(new Serializable[]{StringUtils.substringBefore((String)description, (String)"constituent datasets:") + String.format("constituent datasets:\nPlease see %s for full list of all constituents.", target)}).end().build());
            return dm;
        }
        catch (JAXBException e) {
            throw new InvalidMetadataException("Failed to deserialize datacite xml for DOI " + doi, e);
        }
    }

    public static String truncateDescription(DOI doi, String xml, URI target) throws InvalidMetadataException {
        DataCiteMetadata dm = DataCiteConverter.truncateDescriptionDCM(doi, xml, target);
        return DataCiteValidator.toXml(doi, dm);
    }

    public static String truncateConstituents(DOI doi, String xml, URI target) throws InvalidMetadataException {
        DataCiteMetadata dm = DataCiteConverter.truncateDescriptionDCM(doi, xml, target);
        dm.setRelatedIdentifiers(null);
        return DataCiteValidator.toXml(doi, dm);
    }

    @VisibleForTesting
    protected static String getYear(Date date) {
        if (date == null) {
            return null;
        }
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTime(date);
        return String.valueOf(cal.get(1));
    }

    public static DataCiteMetadata convert(Download d, GbifUser creator, List<DatasetOccurrenceDownloadUsage> usedDatasets, TitleLookup titleLookup) {
        Preconditions.checkNotNull(d.getDoi(), "Download DOI required to build valid DOI metadata");
        Preconditions.checkNotNull(d.getCreated(), "Download created date required to build valid DOI metadata");
        Preconditions.checkNotNull(creator, "Download creator required to build valid DOI metadata");
        Preconditions.checkNotNull(d.getRequest(), "Download request required to build valid DOI metadata");
        DataCiteMetadata.Builder<Void> b = DataCiteMetadata.builder().withIdentifier().withIdentifierType(IdentifierType.DOI.name()).withValue(d.getDoi().getDoiName()).end().withTitles().withTitle(DataCiteMetadata.Titles.Title.builder().withValue(DOWNLOAD_TITLE).build()).end().withSubjects().addSubject().withValue("GBIF").withLang(ENGLISH).end().addSubject().withValue("biodiversity").withLang(ENGLISH).end().addSubject().withValue("species occurrences").withLang(ENGLISH).end().end().withCreators().addCreator().withCreatorName(creator.getName()).end().end().withPublisher(GBIF_PUBLISHER).withPublicationYear(DataCiteConverter.getYear(d.getCreated())).withResourceType().withResourceTypeGeneral(ResourceType.DATASET).end().withAlternateIdentifiers().addAlternateIdentifier().withAlternateIdentifierType("GBIF").withValue(d.getKey()).end().end().withDates().addDate().withDateType(DateType.CREATED).withValue(DataCiteConverter.fdate(d.getCreated())).end().addDate().withDateType(DateType.UPDATED).withValue(DataCiteConverter.fdate(d.getModified())).end().end().withFormats().addFormat(DWCA_FORMAT).end().withSizes().addSize(Long.toString(d.getSize())).end();
        License downloadLicense = d.getLicense() != null && d.getLicense().isConcrete() ? d.getLicense() : DEFAULT_DOWNLOAD_LICENSE;
        b.withRightsList().addRights().withRightsURI(downloadLicense.getLicenseUrl()).withValue(downloadLicense.getLicenseTitle()).end();
        DataCiteMetadata.Descriptions.Description.Builder<DataCiteMetadata.Descriptions.Builder<DataCiteMetadata.Builder<Void>>> db = b.withDescriptions().addDescription().withDescriptionType(DescriptionType.ABSTRACT).withLang(ENGLISH).addContent(new Serializable[]{String.format("A dataset containing %s species occurrences available in GBIF matching the query: %s.", d.getTotalRecords(), DataCiteConverter.getFilterQuery(d, titleLookup))}).addContent(new Serializable[]{String.format("The dataset includes %s records from %s constituent datasets:", d.getTotalRecords(), d.getNumberDatasets())});
        if (!usedDatasets.isEmpty()) {
            DataCiteMetadata.RelatedIdentifiers.Builder<DataCiteMetadata.Builder<Void>> relBuilder = b.withRelatedIdentifiers();
            for (DatasetOccurrenceDownloadUsage du : usedDatasets) {
                if (du.getDatasetDOI() != null) {
                    relBuilder.addRelatedIdentifier().withRelationType(RelationType.REFERENCES).withValue(du.getDatasetDOI().getDoiName()).withRelatedIdentifierType(RelatedIdentifierType.DOI).end();
                }
                if (Strings.isNullOrEmpty(du.getDatasetTitle())) continue;
                db.addContent(new Serializable[]{"\n " + du.getNumberRecords() + " records from " + du.getDatasetTitle() + "."});
            }
            db.addContent(new Serializable[]{LICENSE_INFO});
        }
        return b.build();
    }

    private static DataCiteMetadata.Creators.Creator toDataCiteCreator(Contact contact) {
        DataCiteMetadata.Creators.Creator creator = FACTORY.createDataCiteMetadataCreatorsCreator();
        creator.setCreatorName(ContactAdapter.formatContactName((Contact)contact));
        if (Strings.isNullOrEmpty(creator.getCreatorName())) {
            return null;
        }
        if (!Strings.isNullOrEmpty(contact.getOrganization())) {
            creator.getAffiliation().add(contact.getOrganization());
        }
        for (String userId : contact.getUserId()) {
            DataCiteMetadata.Creators.Creator.NameIdentifier nId = DataCiteConverter.userIdToCreatorNameIdentifier(userId);
            if (nId == null) continue;
            creator.setNameIdentifier(nId);
            break;
        }
        return creator;
    }

    private static DataCiteMetadata.Creators.Creator getDefaultGBIFDataCiteCreator(String fullname) {
        DataCiteMetadata.Creators.Creator creator = FACTORY.createDataCiteMetadataCreatorsCreator();
        creator.setCreatorName(fullname);
        DataCiteMetadata.Creators.Creator.NameIdentifier nid = FACTORY.createDataCiteMetadataCreatorsCreatorNameIdentifier();
        nid.setValue(fullname);
        nid.setSchemeURI("gbif.org");
        nid.setNameIdentifierScheme("GBIF");
        creator.setNameIdentifier(nid);
        return creator;
    }

    private static DataCiteMetadata.Contributors.Contributor toDataCiteContributor(Contact contact) {
        DataCiteMetadata.Contributors.Contributor contributor = FACTORY.createDataCiteMetadataContributorsContributor();
        contributor.setContributorName(ContactAdapter.formatContactName((Contact)contact));
        if (Strings.isNullOrEmpty(contributor.getContributorName())) {
            return null;
        }
        ContributorType contributorType = REGISTRY_DATACITE_ROLE_MAPPING.containsKey(contact.getType()) ? REGISTRY_DATACITE_ROLE_MAPPING.get(contact.getType()) : ContributorType.RELATED_PERSON;
        contributor.setContributorType(contributorType);
        for (String userId : contact.getUserId()) {
            DataCiteMetadata.Contributors.Contributor.NameIdentifier nId = DataCiteConverter.userIdToContributorNameIdentifier(userId);
            if (nId == null) continue;
            contributor.setNameIdentifier(nId);
            break;
        }
        return contributor;
    }

    @VisibleForTesting
    protected static DataCiteMetadata.Creators.Creator.NameIdentifier userIdToCreatorNameIdentifier(String userId) {
        if (Strings.isNullOrEmpty(userId)) {
            return null;
        }
        for (Pattern pattern : SUPPORTED_SCHEMES.keySet()) {
            Matcher matcher = pattern.matcher(userId);
            if (!matcher.matches()) continue;
            DataCiteMetadata.Creators.Creator.NameIdentifier nid = FACTORY.createDataCiteMetadataCreatorsCreatorNameIdentifier();
            nid.setSchemeURI(matcher.group(1));
            nid.setValue(matcher.group(2));
            nid.setNameIdentifierScheme(SUPPORTED_SCHEMES.get(pattern));
            return nid;
        }
        return null;
    }

    protected static DataCiteMetadata.Contributors.Contributor.NameIdentifier userIdToContributorNameIdentifier(String userId) {
        if (Strings.isNullOrEmpty(userId)) {
            return null;
        }
        for (Pattern pattern : SUPPORTED_SCHEMES.keySet()) {
            Matcher matcher = pattern.matcher(userId);
            if (!matcher.matches()) continue;
            DataCiteMetadata.Contributors.Contributor.NameIdentifier nid = FACTORY.createDataCiteMetadataContributorsContributorNameIdentifier();
            nid.setSchemeURI(matcher.group(1));
            nid.setValue(matcher.group(2));
            nid.setNameIdentifierScheme(SUPPORTED_SCHEMES.get(pattern));
            return nid;
        }
        return null;
    }

    private static String getFilterQuery(Download d, TitleLookup titleLookup) {
        try {
            return new HumanFilterBuilder(titleLookup).humanFilterString(d.getRequest().getPredicate());
        }
        catch (Exception ex) {
            return d.getRequest().getPredicate().toString();
        }
    }
}

