/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.common.parsers.geospatial;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import java.util.EnumSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.gbif.api.vocabulary.OccurrenceIssue;
import org.gbif.common.parsers.NumberParser;
import org.gbif.common.parsers.core.OccurrenceParseResult;
import org.gbif.common.parsers.core.ParseResult;
import org.gbif.common.parsers.geospatial.LatLng;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoordinateParseUtils {
    private static final String DMS = "\\s*(\\d{1,3})\\s*[\u00b0d ]\\s*([0-6]?\\d)\\s*['m ]\\s*(?:([0-6]?\\d(?:[,.]\\d+)?)\\s*(?:\"|''|s)?)?\\s*";
    private static final Pattern DMS_SINGLE = Pattern.compile("^\\s*(\\d{1,3})\\s*[\u00b0d ]\\s*([0-6]?\\d)\\s*['m ]\\s*(?:([0-6]?\\d(?:[,.]\\d+)?)\\s*(?:\"|''|s)?)?\\s*$", 2);
    private static final Pattern DMS_COORD = Pattern.compile("^\\s*(\\d{1,3})\\s*[\u00b0d ]\\s*([0-6]?\\d)\\s*['m ]\\s*(?:([0-6]?\\d(?:[,.]\\d+)?)\\s*(?:\"|''|s)?)?\\s*([NSEOW])[ ,;/]?\\s*(\\d{1,3})\\s*[\u00b0d ]\\s*([0-6]?\\d)\\s*['m ]\\s*(?:([0-6]?\\d(?:[,.]\\d+)?)\\s*(?:\"|''|s)?)?\\s*([NSEOW])$", 2);
    private static final String POSITIVE = "NEO";
    private static final Logger LOG = LoggerFactory.getLogger(CoordinateParseUtils.class);

    private CoordinateParseUtils() {
        throw new UnsupportedOperationException("Can't initialize class");
    }

    public static OccurrenceParseResult<LatLng> parseLatLng(String latitude, String longitude) {
        if (Strings.isNullOrEmpty(latitude) || Strings.isNullOrEmpty(longitude)) {
            return OccurrenceParseResult.fail();
        }
        Double lat = NumberParser.parseDouble(latitude);
        Double lng = NumberParser.parseDouble(longitude);
        if (lat == null || lng == null) {
            try {
                lat = CoordinateParseUtils.parseDMS(latitude, true);
                lng = CoordinateParseUtils.parseDMS(longitude, false);
            }
            catch (IllegalArgumentException e) {
                return OccurrenceParseResult.fail(OccurrenceIssue.COORDINATE_INVALID);
            }
        }
        return CoordinateParseUtils.validateAndRound(lat, lng);
    }

    private static boolean inRange(double lat, double lon) {
        return Double.compare(lat, 90.0) <= 0 && Double.compare(lat, -90.0) >= 0 && Double.compare(lon, 180.0) <= 0 && Double.compare(lon, -180.0) >= 0;
    }

    private static boolean isLat(String direction) {
        return "NS".contains(direction.toUpperCase());
    }

    private static int coordSign(String direction) {
        return POSITIVE.contains(direction.toUpperCase()) ? 1 : -1;
    }

    public static OccurrenceParseResult<LatLng> parseVerbatimCoordinates(String coordinates) {
        if (Strings.isNullOrEmpty(coordinates)) {
            return OccurrenceParseResult.fail();
        }
        Matcher m = DMS_COORD.matcher(coordinates);
        if (m.find()) {
            String dir1 = m.group(4);
            String dir2 = m.group(8);
            double c1 = CoordinateParseUtils.coordFromMatcher(m, 1, 2, 3, dir1);
            double c2 = CoordinateParseUtils.coordFromMatcher(m, 5, 6, 7, dir2);
            if (CoordinateParseUtils.isLat(dir1) && !CoordinateParseUtils.isLat(dir2)) {
                return CoordinateParseUtils.validateAndRound(c1, c2);
            }
            if (!CoordinateParseUtils.isLat(dir1) && CoordinateParseUtils.isLat(dir2)) {
                return CoordinateParseUtils.validateAndRound(c2, c1);
            }
            return OccurrenceParseResult.fail(OccurrenceIssue.COORDINATE_INVALID);
        }
        if (coordinates.length() > 4) {
            for (char delim : ",;/ ".toCharArray()) {
                int cnt = StringUtils.countMatches((CharSequence)coordinates, String.valueOf(delim));
                if (cnt != 1) continue;
                String[] latlon = StringUtils.split(coordinates, delim);
                return CoordinateParseUtils.parseLatLng(latlon[0], latlon[1]);
            }
        }
        return OccurrenceParseResult.fail(OccurrenceIssue.COORDINATE_INVALID);
    }

    private static OccurrenceParseResult<LatLng> validateAndRound(double lat, double lon) {
        EnumSet<OccurrenceIssue> issues = EnumSet.noneOf(OccurrenceIssue.class);
        double latOrig = lat;
        double lngOrig = lon;
        lat = CoordinateParseUtils.roundTo5decimals(lat);
        lon = CoordinateParseUtils.roundTo5decimals(lon);
        if (Double.compare(lat, latOrig) != 0 || Double.compare(lon, lngOrig) != 0) {
            issues.add(OccurrenceIssue.COORDINATE_ROUNDED);
        }
        if (Double.compare(lat, 0.0) == 0 && Double.compare(lon, 0.0) == 0) {
            issues.add(OccurrenceIssue.ZERO_COORDINATE);
            return OccurrenceParseResult.success(ParseResult.CONFIDENCE.POSSIBLE, new LatLng(0.0, 0.0), issues);
        }
        if (CoordinateParseUtils.inRange(lat, lon)) {
            return OccurrenceParseResult.success(ParseResult.CONFIDENCE.DEFINITE, new LatLng(lat, lon), issues);
        }
        if ((Double.compare(lat, 90.0) > 0 || Double.compare(lat, -90.0) < 0) && CoordinateParseUtils.inRange(lon, lat)) {
            issues.add(OccurrenceIssue.PRESUMED_SWAPPED_COORDINATE);
            return OccurrenceParseResult.fail(new LatLng(lat, lon), issues);
        }
        issues.add(OccurrenceIssue.COORDINATE_OUT_OF_RANGE);
        return OccurrenceParseResult.fail(issues);
    }

    @VisibleForTesting
    protected static double parseDMS(String coord, boolean lat) {
        String DIRS = lat ? "NS" : "EOW";
        if ((coord = coord.trim().toUpperCase()).length() > 3) {
            char dir = 'n';
            if (DIRS.contains(String.valueOf(coord.charAt(0)))) {
                dir = coord.charAt(0);
                coord = coord.substring(1);
            } else if (DIRS.contains(String.valueOf(coord.charAt(coord.length() - 1)))) {
                dir = coord.charAt(coord.length() - 1);
                coord = coord.substring(0, coord.length() - 1);
            }
            Matcher m = DMS_SINGLE.matcher(coord);
            if (m.find()) {
                return CoordinateParseUtils.coordFromMatcher(m, 1, 2, 3, String.valueOf(dir));
            }
        }
        throw new IllegalArgumentException();
    }

    private static double coordFromMatcher(Matcher m, int idx1, int idx2, int idx3, String sign) {
        return CoordinateParseUtils.roundTo5decimals((double)CoordinateParseUtils.coordSign(sign) * CoordinateParseUtils.dmsToDecimal(NumberParser.parseDouble(m.group(idx1)), NumberParser.parseDouble(m.group(idx2)), NumberParser.parseDouble(m.group(idx3))));
    }

    private static double dmsToDecimal(double degree, Double minutes, Double seconds) {
        minutes = minutes == null ? 0.0 : minutes;
        seconds = seconds == null ? 0.0 : seconds;
        return degree + minutes / 60.0 + seconds / 3600.0;
    }

    private static Double roundTo5decimals(Double x) {
        return x == null ? null : Double.valueOf((double)Math.round(x * Math.pow(10.0, 5.0)) / Math.pow(10.0, 5.0));
    }
}

