/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.utils.file;

import com.google.common.io.Files;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang3.StringUtils;
import org.gbif.utils.collection.CompactHashSet;
import org.gbif.utils.file.InputStreamUtils;
import org.gbif.utils.text.LineComparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    public static final String UTF8 = "UTF8";
    public static final Pattern TAB_DELIMITED = Pattern.compile("\t");
    private static int linesPerMemorySort = 100000;
    private static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);

    public static String classpath2Filepath(String path) {
        return new File(ClassLoader.getSystemResource(path).getFile()).getAbsolutePath();
    }

    public static InputStream classpathStream(String path) throws IOException {
        InputStream in = null;
        URL url = FileUtils.class.getClassLoader().getResource(path);
        if (url != null) {
            in = url.openStream();
        }
        return in;
    }

    public static Set<String> columnsToSet(InputStream source, int ... column) throws IOException {
        return FileUtils.columnsToSet(source, new CompactHashSet<String>(), column);
    }

    public static Set<String> columnsToSet(InputStream source, Set<String> resultSet, int ... column) throws IOException {
        LineIterator lines = FileUtils.getLineIterator(source);
        int maxCols = 0;
        for (int c : column) {
            if (c <= maxCols) continue;
            maxCols = c;
        }
        while (lines.hasNext()) {
            String[] parts;
            String line = lines.nextLine().trim();
            if (FileUtils.ignore(line) || maxCols > (parts = TAB_DELIMITED.split(line)).length) continue;
            for (int c : column) {
                String cell = parts[c].trim();
                resultSet.add(cell);
            }
        }
        return resultSet;
    }

    public static void copyStreams(InputStream in, OutputStream out) throws IOException {
        int bytesRead;
        byte[] buffer = new byte[8192];
        while ((bytesRead = in.read(buffer, 0, 8192)) != -1) {
            out.write(buffer, 0, bytesRead);
        }
        out.close();
        in.close();
    }

    public static void copyStreamToFile(InputStream in, File out) throws IOException {
        FileUtils.copyStreams(in, new FileOutputStream(out));
    }

    public static File createTempDir() throws IOException {
        return FileUtils.createTempDir("gbif-futil", ".tmp");
    }

    public static File createTempDir(String prefix, String suffix) throws IOException {
        File dir = File.createTempFile(prefix, suffix);
        if (!dir.delete()) {
            throw new IOException("Could not delete temp file: " + dir.getAbsolutePath());
        }
        if (!dir.mkdir()) {
            throw new IOException("Could not create temp directory: " + dir.getAbsolutePath());
        }
        return dir;
    }

    public static void deleteDirectoryRecursively(File directory) {
        File[] list;
        for (File file : list = directory.listFiles()) {
            if (file.isDirectory()) {
                FileUtils.deleteDirectoryRecursively(file);
                file.delete();
                continue;
            }
            file.delete();
        }
        directory.delete();
    }

    public static String escapeFilename(String filename) {
        return filename.replaceAll("[\\s./&]", "_");
    }

    public static File getClasspathFile(String path) {
        return new File(ClassLoader.getSystemResource(path).getFile());
    }

    public static InputStream getInputStream(File source) throws FileNotFoundException {
        return new FileInputStream(source);
    }

    public static BufferedReader getInputStreamReader(InputStream input) throws FileNotFoundException {
        return FileUtils.getInputStreamReader(input, UTF8);
    }

    public static BufferedReader getInputStreamReader(InputStream input, String encoding) throws FileNotFoundException {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(input, encoding));
        }
        catch (UnsupportedEncodingException e) {
            LOG.warn("Caught Exception", (Throwable)e);
        }
        return reader;
    }

    public static LineIterator getLineIterator(InputStream source) {
        return FileUtils.getLineIterator(source, UTF8);
    }

    public static LineIterator getLineIterator(InputStream source, String encoding) {
        try {
            return new LineIterator(new BufferedReader(new InputStreamReader(source, encoding)));
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException("Unsupported encoding" + encoding, e);
        }
    }

    public static BufferedReader getUtf8Reader(File file) throws FileNotFoundException {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(file), UTF8));
        }
        catch (UnsupportedEncodingException e) {
            LOG.warn("Caught Exception", (Throwable)e);
        }
        return reader;
    }

    public static String humanReadableByteCount(long bytes, boolean si) {
        int unit;
        int n = unit = si ? 1000 : 1024;
        if (bytes < (long)unit) {
            return bytes + " B";
        }
        int exp = (int)(Math.log(bytes) / Math.log(unit));
        String pre = (si ? "kMGTPE" : "KMGTPE").charAt(exp - 1) + (si ? "" : "i");
        return String.format("%.1f %sB", (double)bytes / Math.pow(unit, exp), pre);
    }

    public static boolean isCompressedFile(File source) {
        String suffix = source.getName().substring(source.getName().lastIndexOf(46) + 1);
        return suffix != null && suffix.length() > 0 && ("zip".equalsIgnoreCase(suffix) || "tgz".equalsIgnoreCase(suffix) || "gz".equalsIgnoreCase(suffix));
    }

    public static ByteBuffer readByteBuffer(File file) throws IOException {
        byte[] content = org.apache.commons.io.FileUtils.readFileToByteArray(file);
        return ByteBuffer.wrap(content);
    }

    public static ByteBuffer readByteBuffer(File file, int bufferSize) throws IOException {
        int b;
        ByteBuffer bbuf = ByteBuffer.allocate(bufferSize);
        BufferedInputStream f = new BufferedInputStream(new FileInputStream(file), bufferSize);
        while ((b = f.read()) != -1 && bbuf.hasRemaining()) {
            bbuf.put((byte)b);
        }
        f.close();
        return bbuf;
    }

    public static void setLinesPerMemorySort(int linesPerMemorySort) {
        FileUtils.linesPerMemorySort = linesPerMemorySort;
    }

    public static Writer startNewUtf8File(File file) throws IOException {
        Files.touch(file);
        return new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file, false), UTF8));
    }

    public static Writer startNewUtf8XmlFile(File file) throws IOException {
        Writer writer = FileUtils.startNewUtf8File(file);
        writer.write("<?xml version='1.0' encoding='utf-8'?>\n");
        return writer;
    }

    public static LinkedList<String> streamToList(InputStream source) throws IOException {
        return FileUtils.streamToList(source, "UTF-8");
    }

    public static List<String> streamToList(InputStream source, List<String> resultList) throws IOException {
        LineIterator lines = FileUtils.getLineIterator(source);
        while (lines.hasNext()) {
            String line = lines.nextLine().trim();
            if (FileUtils.ignore(line)) continue;
            resultList.add(line);
        }
        return resultList;
    }

    public static LinkedList<String> streamToList(InputStream source, String encoding) throws IOException {
        LinkedList<String> resultList = new LinkedList<String>();
        try {
            LineIterator lines = new LineIterator(new BufferedReader(new InputStreamReader(source, encoding)));
            while (lines.hasNext()) {
                String line = lines.nextLine();
                resultList.add(line);
            }
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException("Unsupported encoding " + encoding, e);
        }
        return resultList;
    }

    public static Map<String, String> streamToMap(InputStream source) throws IOException {
        return FileUtils.streamToMap(source, new HashMap<String, String>());
    }

    public static Map<String, String> streamToMap(InputStream source, int key, int value, boolean trimToNull) throws IOException {
        return FileUtils.streamToMap(source, new HashMap<String, String>(), key, value, trimToNull);
    }

    public static Map<String, String> streamToMap(InputStream source, Map<String, String> result) throws IOException {
        LineIterator lines = FileUtils.getLineIterator(source);
        Integer row = 0;
        while (lines.hasNext()) {
            Integer n = row;
            Integer n2 = row = Integer.valueOf(row + 1);
            String line = lines.nextLine().trim();
            if (FileUtils.ignore(line)) continue;
            result.put(line, row.toString());
        }
        return result;
    }

    public static Map<String, String> streamToMap(InputStream source, Map<String, String> result, int key, int value, boolean trimToNull) throws IOException {
        int maxCols;
        LineIterator lines = FileUtils.getLineIterator(source);
        int n = maxCols = key > value ? key : value + 1;
        while (lines.hasNext()) {
            String[] parts;
            String line = lines.nextLine();
            if (FileUtils.ignore(line) || maxCols > (parts = TAB_DELIMITED.split(line)).length) continue;
            if (trimToNull) {
                result.put(StringUtils.trimToNull(parts[key]), StringUtils.trimToNull(parts[value]));
                continue;
            }
            result.put(parts[key], parts[value]);
        }
        return result;
    }

    public static Set<String> streamToSet(InputStream source) throws IOException {
        return FileUtils.streamToSet(source, new CompactHashSet<String>());
    }

    public static Set<String> streamToSet(InputStream source, Set<String> resultSet) throws IOException {
        LineIterator lines = FileUtils.getLineIterator(source);
        while (lines.hasNext()) {
            String line = lines.nextLine().trim();
            if (FileUtils.ignore(line)) continue;
            resultSet.add(line);
        }
        return resultSet;
    }

    public static String toFilePath(URL url) {
        String protocol = url.getProtocol() == null || "http".equalsIgnoreCase(url.getProtocol()) ? "" : "/__" + url.getProtocol() + "__";
        String domain = url.getAuthority() == null ? "__domainless" : url.getAuthority();
        return domain + protocol + url.getFile();
    }

    public static File url2file(URL url) {
        File f = null;
        try {
            f = new File(url.toURI());
        }
        catch (URISyntaxException e) {
            f = new File(url.getPath());
        }
        return f;
    }

    static int lowestValueIndex(List<String> values, Comparator<String> comparator) {
        int index = 0;
        String lowestValue = null;
        for (int i = 0; i < values.size(); ++i) {
            String value = values.get(i);
            if (lowestValue == null) {
                lowestValue = value;
                index = i;
                continue;
            }
            if (comparator.compare(lowestValue, value) <= 0) continue;
            lowestValue = value;
            index = i;
        }
        return lowestValue == null ? -1 : index;
    }

    private static File getChunkFile(File original, int index) {
        return new File(original.getParentFile(), FilenameUtils.getBaseName(original.getName()) + '_' + index + Files.getFileExtension(original.getName()));
    }

    private static boolean ignore(String line) {
        return StringUtils.trimToNull(line) == null || line.startsWith("#");
    }

    public int getLinesPerMemorySort() {
        return linesPerMemorySort;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void mergedSortedFiles(List<File> sortFiles, FileWriter sortedFileWriter, Comparator<String> lineComparator) throws IOException {
        LinkedList<BufferedReader> partReaders = new LinkedList<BufferedReader>();
        try {
            String partLine;
            LinkedList<String> partReaderLine = new LinkedList<String>();
            for (File file : sortFiles) {
                partReaders.add(new BufferedReader(new FileReader(file)));
            }
            boolean moreData = false;
            for (BufferedReader partReader : partReaders) {
                partLine = partReader.readLine();
                if (partLine != null) {
                    moreData = true;
                }
                partReaderLine.add(partLine);
            }
            while (moreData) {
                int runtimeException = FileUtils.lowestValueIndex(partReaderLine, lineComparator);
                if (runtimeException >= 0) {
                    sortedFileWriter.write((String)partReaderLine.get(runtimeException));
                    sortedFileWriter.write("\n");
                    BufferedReader r = (BufferedReader)partReaders.get(runtimeException);
                    partLine = r.readLine();
                    LinkedList<String> linkedList = partReaderLine;
                    synchronized (linkedList) {
                        partReaderLine.add(runtimeException, partLine);
                        partReaderLine.remove(runtimeException + 1);
                        continue;
                    }
                }
                moreData = false;
            }
        }
        finally {
            for (BufferedReader b : partReaders) {
                try {
                    b.close();
                }
                catch (RuntimeException runtimeException) {}
            }
            sortedFileWriter.flush();
            sortedFileWriter.close();
            for (File f : sortFiles) {
                f.delete();
            }
        }
    }

    public void sort(File input, File sorted, String encoding, int column, String columnDelimiter, Character enclosedBy, String newlineDelimiter, int ignoreHeaderLines) throws IOException {
        LineComparator lineComparator = enclosedBy == null ? new LineComparator(column, columnDelimiter) : new LineComparator(column, columnDelimiter, enclosedBy);
        this.sort(input, sorted, encoding, column, columnDelimiter, enclosedBy, newlineDelimiter, ignoreHeaderLines, lineComparator, false);
    }

    public void sort(File input, File sorted, String encoding, int column, String columnDelimiter, Character enclosedBy, String newlineDelimiter, int ignoreHeaderLines, Comparator<String> lineComparator, boolean ignoreCase) throws IOException {
        LOG.debug("sorting " + input.getAbsolutePath() + " as new file " + sorted.getAbsolutePath());
        if (encoding == null) {
            LOG.warn("No encoding specified, assume UTF-8");
            encoding = "UTF-8";
        }
        if (!this.sortInUnix(input, sorted, encoding, ignoreHeaderLines, column, columnDelimiter, newlineDelimiter, ignoreCase)) {
            LOG.debug("No unix sort available, using native java sorting");
            this.sortInJava(input, sorted, encoding, lineComparator, ignoreHeaderLines);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void sortInJava(File input, File sorted, String encoding, Comparator<String> lineComparator, int ignoreHeaderLines) throws IOException {
        LOG.debug("Sorting File[" + input.getAbsolutePath() + ']');
        long start = System.currentTimeMillis();
        LinkedList<File> sortFiles = new LinkedList<File>();
        LinkedList<String> headerLines = new LinkedList<String>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(input), encoding));){
            String line = br.readLine();
            int fileCount = 0;
            LinkedList<String> linesToSort = new LinkedList<String>();
            while (line != null) {
                if (ignoreHeaderLines > 0) {
                    headerLines.add(line);
                    --ignoreHeaderLines;
                } else {
                    linesToSort.add(line);
                    if (linesToSort.size() == linesPerMemorySort) {
                        sortFiles.add(this.sortAndWrite(input, encoding, lineComparator, fileCount, linesToSort));
                        linesToSort = new LinkedList();
                        ++fileCount;
                    }
                }
                line = br.readLine();
            }
            if (!linesToSort.isEmpty()) {
                sortFiles.add(this.sortAndWrite(input, encoding, lineComparator, fileCount, linesToSort));
            }
        }
        LOG.debug(sortFiles.size() + " sorted file chunks created in " + (System.currentTimeMillis() - start) / 1000L + " secs");
        FileWriter sortedFileWriter = new FileWriter(sorted);
        for (String h : headerLines) {
            sortedFileWriter.write(h);
            sortedFileWriter.write("\n");
        }
        this.mergedSortedFiles(sortFiles, sortedFileWriter, lineComparator);
        LOG.debug("File " + input.getAbsolutePath() + " sorted successfully using " + sortFiles.size() + " parts to do sorting in " + (System.currentTimeMillis() - start) / 1000L + " secs");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<File> split(File input, int linesPerOutput, String extension) throws IOException {
        LOG.debug("Splitting File[" + input.getAbsolutePath() + ']');
        long timer = System.currentTimeMillis();
        LinkedList<File> splitFiles = new LinkedList<File>();
        BufferedReader br = new BufferedReader(new FileReader(input));
        String line = br.readLine();
        int fileCount = 0;
        File splitFile = FileUtils.getChunkFile(input, fileCount);
        ++fileCount;
        splitFiles.add(splitFile);
        try (FileWriter fw = new FileWriter(splitFile);){
            int lineCount = 0;
            while (line != null) {
                if (lineCount == linesPerOutput) {
                    fw.flush();
                    fw.close();
                    splitFile = FileUtils.getChunkFile(input, fileCount);
                    splitFiles.add(splitFile);
                    fw = new FileWriter(splitFile);
                    ++fileCount;
                    lineCount = 0;
                }
                fw.write(line);
                fw.write("\n");
                line = br.readLine();
                ++lineCount;
            }
            fw.flush();
        }
        LOG.debug("File[" + input.getAbsolutePath() + "] split successfully into[" + splitFiles.size() + "] parts in secs[" + (1L + System.currentTimeMillis() - timer) / 1000L + "]");
        return splitFiles;
    }

    protected boolean sortInUnix(File input, File sorted, String encoding, int ignoreHeaderLines, int column, String columnDelimiter, String lineDelimiter, boolean ignoreCase) throws IOException {
        if (column != 0 || lineDelimiter == null || !lineDelimiter.contains("\n") || columnDelimiter != null && columnDelimiter.contains("\n")) {
            LOG.debug("Cannot use unix sort on this file");
            return false;
        }
        boolean success = false;
        try {
            Process process;
            String command;
            LinkedList<String> cmds = new LinkedList<String>();
            cmds.add("/bin/sh");
            cmds.add("-c");
            cmds.add("");
            ProcessBuilder pb = new ProcessBuilder(cmds);
            Map<String, String> env = pb.environment();
            env.clear();
            env.put("LC_ALL", "C");
            if (ignoreHeaderLines > 0) {
                command = "head -n " + ignoreHeaderLines + ' ' + input.getAbsolutePath() + " > " + sorted.getAbsolutePath();
                LOG.debug("Issue unix sort cmd: " + command);
                cmds.removeLast();
                cmds.add(command);
                process = pb.start();
                process.waitFor();
                command = "sed " + ignoreHeaderLines + "d " + input.getAbsolutePath() + " | sort " + (ignoreCase ? "--ignore-case" : "") + " >> " + sorted.getAbsolutePath();
            } else {
                command = "sort -o " + sorted.getAbsolutePath() + ' ' + input.getAbsolutePath();
            }
            LOG.debug("Issue unix sort cmd: " + command);
            cmds.removeLast();
            cmds.add(command);
            process = pb.start();
            InputStream err = process.getErrorStream();
            int exitValue = process.waitFor();
            if (exitValue == 0) {
                LOG.debug("Successfully sorted file with unix sort");
                success = true;
            } else {
                LOG.warn("Error sorting file with unix sort");
                InputStreamUtils isu = new InputStreamUtils();
                System.err.append(isu.readEntireStream(err));
            }
        }
        catch (Exception e) {
            LOG.warn("Caught Exception", (Throwable)e);
        }
        return success;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File sortAndWrite(File input, String encoding, Comparator<String> lineComparator, int fileCount, List<String> linesToSort) throws IOException {
        long start = System.currentTimeMillis();
        Collections.sort(linesToSort, lineComparator);
        LOG.debug("Collections.sort took msec[" + (System.currentTimeMillis() - start) + "] to sort records[" + linesToSort.size() + ']');
        File sortFile = FileUtils.getChunkFile(input, fileCount);
        try (OutputStreamWriter fw = new OutputStreamWriter((OutputStream)new FileOutputStream(sortFile), encoding);){
            for (String s : linesToSort) {
                fw.write(s);
                fw.write("\n");
            }
        }
        return sortFile;
    }
}

