/*
 * Decompiled with CFR 0.152.
 */
package org.gbif.utils.file;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressionUtil {
    private static final Logger LOG = LoggerFactory.getLogger(CompressionUtil.class);
    private static final int BUFFER = 2048;
    private static final String APPLE_RESOURCE_FORK = "__MACOSX";

    private CompressionUtil() {
        throw new UnsupportedOperationException("Can't initialize class");
    }

    public static File decompressFile(File compressedFile) throws IOException, UnsupportedCompressionType {
        File dir = File.createTempFile("gbif-", null);
        if (dir.exists() && !dir.delete()) {
            throw new IOException("Couldn't delete temporary directory");
        }
        if (!dir.mkdirs()) {
            throw new IOException("Couldn't create temporary directory for decompression");
        }
        CompressionUtil.decompressFile(dir, compressedFile);
        return dir;
    }

    public static List<File> decompressFile(File directory, File compressedFile) throws IOException, UnsupportedCompressionType {
        return CompressionUtil.decompressFile(directory, compressedFile, false);
    }

    public static List<File> decompressFile(File directory, File compressedFile, boolean keepSubdirectories) throws IOException, UnsupportedCompressionType {
        List<File> files = null;
        try {
            files = CompressionUtil.unzipFile(directory, compressedFile, keepSubdirectories);
        }
        catch (ZipException e) {
            LOG.debug("No zip compression");
        }
        if (files == null) {
            try {
                files = CompressionUtil.ungzipFile(directory, compressedFile);
            }
            catch (Exception e) {
                LOG.debug("No gzip compression");
                throw new UnsupportedCompressionType("Unknown compression type. Neither zip nor gzip", e);
            }
        }
        return files;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<File> ungzipFile(File directory, File zipFile) throws IOException {
        ArrayList<File> files = new ArrayList<File>();
        try (TarArchiveInputStream in = new TarArchiveInputStream((InputStream)new GZIPInputStream(new FileInputStream(zipFile)));){
            TarArchiveEntry entry = in.getNextTarEntry();
            while (entry != null) {
                if (entry.isDirectory()) {
                    LOG.debug("TAR archive contains directories which are being ignored");
                    entry = in.getNextTarEntry();
                    continue;
                }
                String fn = new File(entry.getName()).getName();
                if (fn.startsWith(".")) {
                    LOG.debug("TAR archive contains a hidden file which is being ignored");
                    entry = in.getNextTarEntry();
                    continue;
                }
                File targetFile = new File(directory, fn);
                if (targetFile.exists()) {
                    LOG.warn("TAR archive contains duplicate filenames, only the first is being extracted");
                    entry = in.getNextTarEntry();
                    continue;
                }
                LOG.debug("Extracting file: {} to: {}", (Object)entry.getName(), (Object)targetFile.getAbsolutePath());
                FileOutputStream out = new FileOutputStream(targetFile);
                try {
                    IOUtils.copy((InputStream)in, (OutputStream)out);
                    out.close();
                }
                finally {
                    IOUtils.closeQuietly(out);
                }
                files.add(targetFile);
            }
        }
        return files;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<File> ungzipFile(File directory, File zipFile, boolean isTarred) throws IOException {
        if (isTarred) {
            return CompressionUtil.ungzipFile(directory, zipFile);
        }
        ArrayList<File> files = new ArrayList<File>();
        GZIPInputStream in = null;
        BufferedOutputStream dest = null;
        try {
            int count;
            in = new GZIPInputStream(new FileInputStream(zipFile));
            String unzippedName = zipFile.getName().substring(0, zipFile.getName().lastIndexOf("."));
            File outputFile = new File(directory, unzippedName);
            LOG.debug("Extracting file: {} to: {}", (Object)unzippedName, (Object)outputFile.getAbsolutePath());
            FileOutputStream fos = new FileOutputStream(outputFile);
            dest = new BufferedOutputStream(fos, 2048);
            byte[] data = new byte[2048];
            while ((count = in.read(data, 0, 2048)) != -1) {
                dest.write(data, 0, count);
            }
            files.add(outputFile);
        }
        finally {
            if (in != null) {
                in.close();
            }
            if (dest != null) {
                dest.flush();
                dest.close();
            }
        }
        return files;
    }

    public static List<File> unzipFile(File directory, File zipFile) throws IOException {
        return CompressionUtil.unzipFile(directory, zipFile, false);
    }

    public static void zipDir(File dir, File zipFile) throws IOException {
        CompressionUtil.zipDir(dir, zipFile, false);
    }

    public static void zipDir(File dir, File zipFile, boolean inclSubdirs) throws IOException {
        Collection<File> files = FileUtils.listFiles(dir, null, inclSubdirs);
        CompressionUtil.zipFiles(files, dir, zipFile);
    }

    public static void zipFile(File file, File zipFile) throws IOException {
        HashSet<File> files = new HashSet<File>();
        files.add(file);
        CompressionUtil.zipFiles(files, file.getParentFile(), zipFile);
    }

    public static void zipFiles(Collection<File> files, File rootContext, File zipFile) throws IOException {
        if (files.isEmpty()) {
            LOG.info("no files to zip.");
        } else {
            try {
                BufferedInputStream origin = null;
                FileOutputStream dest = new FileOutputStream(zipFile);
                ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(dest));
                byte[] data = new byte[2048];
                for (File f : files) {
                    int count;
                    LOG.debug("Adding file {} to archive", (Object)f);
                    FileInputStream fi = new FileInputStream(f);
                    origin = new BufferedInputStream(fi, 2048);
                    String zipPath = StringUtils.removeStart((String)f.getAbsolutePath(), (String)(rootContext.getAbsolutePath() + File.separator));
                    ZipEntry entry = new ZipEntry(zipPath);
                    out.putNextEntry(entry);
                    while ((count = origin.read(data, 0, 2048)) != -1) {
                        out.write(data, 0, count);
                    }
                    origin.close();
                }
                out.finish();
                out.close();
            }
            catch (IOException e) {
                LOG.error("IOException while zipping files: {}", files);
                throw e;
            }
        }
    }

    public static List<File> unzipFile(File directory, File zipFile, boolean keepSubdirectories) throws IOException {
        LOG.debug("Unzipping archive " + zipFile.getName() + " into directory: " + directory.getAbsolutePath());
        ZipFile zf = new ZipFile(zipFile);
        Enumeration<? extends ZipEntry> entries = zf.entries();
        while (entries.hasMoreElements()) {
            ZipEntry entry = entries.nextElement();
            if (entry.getName().toUpperCase().contains(APPLE_RESOURCE_FORK)) {
                LOG.debug("Ignoring resource fork file: " + entry.getName());
                continue;
            }
            if (entry.isDirectory()) {
                if (CompressionUtil.isHiddenFile(new File(entry.getName()))) {
                    LOG.debug("Ignoring hidden directory: " + entry.getName());
                    continue;
                }
                if (keepSubdirectories) {
                    new File(directory, entry.getName()).mkdir();
                    continue;
                }
                LOG.debug("Ignoring (sub)directory: " + entry.getName());
                continue;
            }
            if (CompressionUtil.isHiddenFile(new File(entry.getName()))) {
                LOG.debug("Ignoring hidden file: " + entry.getName());
                continue;
            }
            File targetFile = keepSubdirectories ? new File(directory, entry.getName()) : new File(directory, new File(entry.getName()).getName());
            CompressionUtil.createParentFolder(targetFile);
            CompressionUtil.extractFile(zf, entry, targetFile);
        }
        zf.close();
        if (keepSubdirectories) {
            CompressionUtil.removeRootDirectory(directory);
        }
        return directory.listFiles() == null ? new ArrayList() : Arrays.asList(directory.listFiles());
    }

    private static boolean isHiddenFile(File f) {
        if (f.getName().startsWith(".")) {
            return true;
        }
        if (f.getParentFile() != null) {
            return CompressionUtil.isHiddenFile(f.getParentFile());
        }
        return false;
    }

    private static void removeRootDirectory(File directory) {
        File root;
        File[] rootFiles = directory.listFiles(HiddenFileFilter.VISIBLE);
        if (rootFiles.length == 1 && (root = rootFiles[0]).isDirectory()) {
            LOG.debug("Removing single root folder {} found in decompressed archive", (Object)root.getAbsoluteFile());
            for (File f : FileUtils.listFilesAndDirs(root, TrueFileFilter.TRUE, TrueFileFilter.TRUE)) {
                File f2 = new File(directory, f.getName());
                f.renameTo(f2);
            }
            root.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void extractFile(ZipFile zf, ZipEntry entry, File targetFile) {
        try {
            LOG.debug("Extracting file: {} to: {}", (Object)entry.getName(), (Object)targetFile.getAbsolutePath());
            InputStream in = zf.getInputStream(entry);
            BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(targetFile));
            try {
                IOUtils.copy(zf.getInputStream(entry), (OutputStream)out);
            }
            finally {
                in.close();
                ((OutputStream)out).close();
            }
        }
        catch (IOException e) {
            LOG.error("File could not be extraced: " + e.getMessage(), (Throwable)e);
        }
    }

    private static void createParentFolder(File file) {
        File parent = new File(file.getParent());
        if (!parent.exists()) {
            LOG.debug(parent.mkdirs() ? "Created parent directory: " + parent.getAbsolutePath() : "Failed to create parent directory: " + parent.getAbsolutePath());
        }
    }

    public static class UnsupportedCompressionType
    extends RuntimeException {
        public UnsupportedCompressionType() {
        }

        public UnsupportedCompressionType(String message) {
            super(message);
        }

        public UnsupportedCompressionType(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

